import configparser
import json
import os
import threading
import time

import schedule
import yaml
from watchdog.events import FileSystemEventHandler
from watchdog.observers import Observer

from scripts.constants.app_config import app_config
from scripts.constants.app_constants import constants, services
from scripts.constants.app_variables import DeviceInfo
from scripts.constants.events import events_constants
from scripts.handlers.redundancy_handler import RedundancyHandler
from scripts.logging.logger import logger
from scripts.utilities.common_util import common_utilities
from scripts.utilities.communication_util import post_events, post_data
from scripts.utilities.mqtt_subscription import MQTTSub
from scripts.utilities.service_util import service_operations
from scripts.utilities.udp_subscription import UDPSub

conf_file_path = os.path.join(constants.ilens_agent_path, constants.conf_path, "application.conf")
try:
    DeviceInfo.device_id, DeviceInfo.project_id = common_utilities.fetch_device_details_from_lic(
        constants.license_file_path, constants.secret_key
    )
    logger.info(f"device_id: \'{DeviceInfo.device_id}\' and project_id: \'{DeviceInfo.project_id}\'")
    if not DeviceInfo.device_id or not DeviceInfo.project_id:
        DeviceInfo.device_id, DeviceInfo.project_id = common_utilities.fetch_data_from_conf(conf_file_path)
        logger.info("Reading \'device_id\' and \'project_id\' from application conf"
                    " file due to error in loading data from license file.")
except Exception as e:
    DeviceInfo.device_id, DeviceInfo.project_id = common_utilities.fetch_data_from_conf(conf_file_path)
    logger.info(f"device_id: \'{DeviceInfo.device_id}\' and project_id: \'{DeviceInfo.project_id}\' - {e}")


class FilesHandler:
    @staticmethod
    def daily_sync():
        response_config_data = {}
        try:
            logger.info("Entered in daily_sync")
            files_list = app_config.files_to_watch
            for each_file in files_list:
                for filename in constants.file_names:
                    file_name = os.path.join(each_file, filename)
                    if os.path.isfile(file_name):
                        file_data = common_utilities.read_configs(file_name)
                        response_config_data[file_name] = file_data
            post_data(response_config_data, endpoint="sync_config")
            post_events(event_code=events_constants.daily_sync)
            logger.info("Exited from daily_sync")
        except Exception as e:
            logger.exception(f"Exception occurred while running daily sync - {e}.")

    @staticmethod
    def sync_config(files_path, file_content):
        try:
            logger.info("Entered in sync_config")
            logger.info(f"filepath - {files_path} and file_content - {file_content}")
            check_pipeline = False
            if files_path.endswith(".conf"):
                config = configparser.ConfigParser()
                config.read_string(file_content)
                config['AGENT']['agent_id'] = DeviceInfo.device_id
                config['AGENT']['registration_project_id'] = DeviceInfo.project_id
                config['MANAGER']['base_url'] = app_config.agent_base_url
                updated_config_string = ''
                for section in config.sections():
                    updated_config_string += f'[{section}]\n'
                    for key, value in config.items(section):
                        updated_config_string += f'{key} = {value}\n'
                    updated_config_string += '\n'
                file_data = updated_config_string
            elif files_path.endswith("channel.yml"):
                uploader_list = [
                    node for node in file_content.get("flow_data", {}).get("nodes", {}).keys()
                    if not node.startswith("device_instance_") if not node.startswith("collector_")
                ]
                for uploader in uploader_list:
                    received_node_host = file_content["flow_data"]["nodes"][uploader]["node_configuration"]["host"]
                    if received_node_host not in ["localhost", "127.0.0.1"]:
                        print(
                            f"received host: {file_content['flow_data']['nodes'][uploader]['node_configuration']['host']}")
                        file_content['flow_data']['nodes'][
                            uploader]['node_configuration']['host'] = app_config.channel_uploader_host
                        file_content['flow_data']['nodes'][
                            uploader]['node_configuration']['port'] = app_config.channel_uploader_port
                file_data = json.dumps(file_content)
                check_pipeline = True
            else:
                if "monitoring_engine" in files_path:
                    if (file_content["uploader"]["type"]).lower() == "http":
                        if app_config.monitoring_engine_url:
                            file_content["uploader"]["url"] = app_config.monitoring_engine_url
                        else:
                            logger.critical("Error while updating monitoring engine's  configuration file,"
                                            " check the field - url in the redundancy configuration.")
                    elif (file_content["uploader"]["type"]).lower() == "udp":
                        if app_config.monitoring_engine_host and app_config.monitoring_engine_port:
                            file_content["uploader"]["host"] = app_config.monitoring_engine_host
                            file_content["uploader"]["port"] = app_config.monitoring_engine_port
                        else:
                            logger.critical("Error while updating monitoring engine's configuration file,"
                                            " check the fields - host_ip and port in the redundancy configuration.")
                file_data = yaml.dump(file_content, default_flow_style=False)
            common_utilities.update_file(files_path, file_data)
            event_code = events_constants.pipeline_updated if check_pipeline else events_constants.configuration_updated
            post_events(event_code=event_code)
            service_name = service_operations.service_name_mapper(files_path)
            if service_name != services.acquisition_engine:
                service_status = service_operations.restart_service(service_name)
                if service_status:
                    post_events(event_code=events_constants.secondary_module_restarted, module_name=service_name)
                    print("Service restarted successfully.")
                    logger.info("Service restarted successfully.")
                else:
                    post_events(event_code=events_constants.secondary_module_restart_failed)
                    print(f"Failed to restart the module - {service_name}.")
                    logger.info(f"Failed to restart the module - {service_name}.")
            logger.info("Exited from sync_config")
        except Exception as e:
            logger.exception(f"Exception occurred while syncing configuration file - {e}.")

    @staticmethod
    def is_file_path_need(files_path):
        try:
            logger.info("Entered in is_file_path_need")
            logger.info(f"filepath - {files_path}")
            if constants.file_path_changes:
                from scripts.utilities.common_util import WindowsUtilities

                modified_path = WindowsUtilities().modify_file_path(files_path)
                logger.info("Exiting from is_file_path_need")
                return modified_path

            else:
                logger.info("Exiting from is_file_path_need")
                return files_path
        except Exception as e:
            logger.exception(f"Exception occurred - {e}.")

        return None


class FileChangeHandler(FileSystemEventHandler):
    def on_modified(self, event):
        transfer_config(event)

    def on_created(self, event):
        transfer_config(event)

    def on_deleted(self, event):
        logger.critical(f"File {event.src_path} has been deleted.")

    def on_moved(self, event):
        if any(event.dest_path.endswith(extension) for extension in constants.file_names):
            transfer_config(event, check_destination=True)


def transfer_config(event, check_destination=False):
    logger.info("Entered in transfer_config")
    logger.info(f"{event} - {check_destination}")
    if not event.is_directory:
        event_code = (
            events_constants.pipeline_modification
            if event.src_path.endswith(constants.pipeline_file)
            else events_constants.configuration_modification
        )
        if any(event.src_path.endswith(extension) for extension in constants.file_names):
            file_data = common_utilities.read_configs(event.src_path)
            response_json = {
                "file_name": event.src_path,
                "data": file_data
            }
            post_data(response_json)
            post_events(event_code)
            logger.info(f"File {event.src_path} has been modified.")
        elif check_destination:
            file_data = common_utilities.read_configs(event.src_path)
            response_json = {
                "file_name": event.dest_path,
                "data": file_data
            }
            post_data(response_json)
            post_events(event_code)
            logger.info(f"File {event.dest_path} has been modified.")
    logger.info("Exiting from transfer_config")


def redundancy_initializer():
    try:
        logger.info("Entered in redundancy_initializer")
        file_handler = FilesHandler()
        event_handler = FileChangeHandler()
        redundancy_handler = RedundancyHandler()

        if app_config.is_data_source:
            client_status = service_operations.check_ilens_client_status()
            if not client_status:
                if service_operations.restart_service(services.acquisition_engine):
                    logger.info("Acquisition Engine started on running primary redundancy module.")
            if app_config.local_uploader_type.lower() == "mqtt":
                logger.info("Using local MQTT Subscriber.")
                MQTTSub(app_config.local_uploader_ip, app_config.local_uploader_port, app_config.local_mqtt_topic)
            elif app_config.local_uploader_type.lower() == "udp":
                logger.info("Using local UDP Subscriber.")
                local_udp_thread = threading.Thread(
                    target=UDPSub,
                    args=(app_config.local_uploader_ip, app_config.local_uploader_port, 1024000)
                )
                local_udp_thread.start()

        schedule.every().day.at(app_config.resync_time).do(file_handler.daily_sync)

        observer = Observer()

        for files_path in app_config.files_to_watch:
            observer.schedule(event_handler, files_path, recursive=True)

        observer.start()

        try:
            while True:
                schedule.run_pending()
                post_data(redundancy_handler.fetch_device_details(), endpoint="fetch_device_details")
                time.sleep(app_config.run_time)
        except KeyboardInterrupt:
            observer.stop()
        observer.join()
    except Exception as e:
        logger.exception(f"Exception occurred while monitoring files - {e}")
