import pandas as pd
from loguru import logger
import warnings
warnings.filterwarnings('ignore')

base_path = f'data'


class BatchMaster:
    def __init__(self):
        self.sheet = "P2E Golden batch 05-01-2023.xlsx"

    def read_file(self, sheet_name, skiprows=1):
        try:
            df = pd.read_excel(f'{base_path}/{self.sheet}', sheet_name=sheet_name,
                               skiprows=skiprows)
            return df
        except Exception as e:
            logger.exception(f"Exception - {e}")

    def read_df(self):
        try:
            df_stage_1 = self.read_file(sheet_name="Stage I")
            df_stage_2 = self.read_file(sheet_name="Stage II")
            df_stage_3 = self.read_file(sheet_name="Stage III")
            df_stage_4 = self.read_file(sheet_name="Stage IV")
            df_stage_5 = self.read_file(sheet_name="Stage V")
            logger.debug(f'{df_stage_1.shape}, {df_stage_2.shape}, {df_stage_3.shape}, {df_stage_4.shape},'
                         f'{df_stage_5.shape},')

            return df_stage_1, df_stage_2, df_stage_3, df_stage_4, df_stage_5
        except Exception as e:
            logger.exception(f"Exception - {e}")

    @staticmethod
    def preprocess_master_df_proto(df, process_stage_name):
        try:
            df.rename(columns={'Batch  Number': 'batch_no', 'Start Time.1': 'batch_start_time',
                               'End Time.1': 'batch_end_time', 'Downtime': 'downtime',
                               'Equipment': 'used_equipment'}, inplace=True)
            df_stage = pd.DataFrame(index=[i for i in range(len(df))])
            if process_stage_name == "Stage-01":
                df_stage['batch_no'] = df['batch_no']
                df_stage['input_qty'] = df['Input Qty. (kg) F00056-BULK-001']
                df_stage['formaldehyde_content'] = df['F00041-BULK-001 content nil']
                df_stage['unreactive_alpha_picoline'] = df['un reacted F00056-BULK-001 ( for infor.)']
                df_stage['moisture_content'] = df['Moisture content \n(for infor)']

            elif process_stage_name == "Stage-02":
                df_stage['batch_no'] = df['batch_no']
                df_stage['input_qty'] = df['Input Qty.\n(kg) Stage-I']
                df_stage['formaldehyde_content'] = None
                df_stage['unreactive_alpha_picoline'] = df['F00056-BULK-001 \n(NLT 50 %)']
                df_stage['moisture_content'] = df['Moisture content \n(for Information).1']

            elif process_stage_name == "Stage-03":
                df_stage['batch_no'] = df['batch_no']
                df_stage['input_qty'] = df['Input Qty. (kg) Stage-I']
                df_stage['formaldehyde_content'] = None
                df_stage['unreactive_alpha_picoline'] = None
                df_stage['moisture_content'] = df['M/C for infor']

            elif process_stage_name == "Stage-04":
                df_stage['batch_no'] = df['batch_no']
                df_stage['input_qty'] = None
                df_stage['formaldehyde_content'] = None
                df_stage['unreactive_alpha_picoline'] = None
                df_stage['moisture_content'] = df['NMT 0.5%']

            elif process_stage_name == "Stage-05":
                df_stage['batch_no'] = df['batch_no']
                df_stage['input_qty'] = None
                df_stage['formaldehyde_content'] = None
                df_stage['unreactive_alpha_picoline'] = None
                df_stage['moisture_content'] = df[' NMT 1.0%']
            df_stage.reset_index(drop=True, inplace=True)
            df_stage = df_stage[['batch_no', 'input_qty', 'formaldehyde_content',
                                 'unreactive_alpha_picoline', 'moisture_content']]

            return df_stage
        except Exception as e:
            logger.exception(f"Exception - {e}")

    @staticmethod
    def preprocess_master_df(df, process_stage_name, batch_product, ideal_batch_cycle_time_hr, batch_setup_time_hr,
                             process_stage_id, work_order_no="P2E-STD-01", work_order_item_no="P2E-STD-01-01",
                             final_product="P2E-Stage-05"):
        try:

            df = df[['Batch  Number', 'Start Time.1', 'End Time.1', 'Downtime', 'Equipment']]
            df.rename(columns={'Batch  Number': 'batch_no', 'Start Time.1': 'batch_start_time',
                               'End Time.1': 'batch_end_time', 'Downtime': 'downtime',
                               'Equipment': 'used_equipment'}, inplace=True)
            df['process_stage_name'] = process_stage_name
            df['process_stage_id'] = process_stage_id
            df['work_order_no'] = work_order_no
            df['work_order_item_no'] = work_order_item_no
            df['final_product'] = final_product
            df['batch_product'] = batch_product
            df['ideal_batch_cycle_time_hr'] = ideal_batch_cycle_time_hr
            df['batch_cycle_time'] = df['batch_end_time'] - df['batch_start_time']
            df['batch_cycle_time_minutes'] = df['batch_cycle_time'].dt.total_seconds() // 60
            df['batch_setup_time_hr'] = batch_setup_time_hr
            df['selected_equipments'] = df['used_equipment']
            df['created_on'] = df['batch_start_time'].dt.date
            df['created_by'] = 'Aakash'
            df['last_updated_on'] = df['batch_end_time'].dt.date
            df['last_updated_by'] = 'Aakash'
            df = df.astype({'batch_cycle_time': str})

            df = df[['batch_no', 'final_product', 'batch_product', 'process_stage_name', 'process_stage_id',
                     'work_order_no', 'work_order_item_no', 'ideal_batch_cycle_time_hr', 'batch_start_time',
                     'batch_end_time', 'batch_cycle_time', 'batch_cycle_time_minutes', 'batch_setup_time_hr',
                     'downtime', 'selected_equipments', 'created_on', 'created_by', 'last_updated_on',
                     'last_updated_by']]
            df.reset_index(drop=True, inplace=True)
            return df
        except Exception as e:
            logger.exception(f"Exception - {e}")

    @staticmethod
    def preprocess_kpi_df(df, kpi, process_stage_name, process_stage_id):
        try:
            if kpi == "Quality":
                df = df[['Batch  Number', 'Start Time.1', 'End Time.1', 'Quality']]
                df.rename(columns={'Batch  Number': 'batch_no', 'Start Time.1': 'batch_start_time',
                                   'End Time.1': 'batch_end_time', 'Quality': 'kpi_value'}, inplace=True)
            df['process_stage_name'] = process_stage_name
            df['process_stage_id'] = process_stage_id
            df['kpi_name'] = kpi
            df['kpi_description'] = f'{kpi} of this batch'
            return df
        except Exception as e:
            logger.exception(f"Exception - {e}")

    def orchestrator_kpi(self):
        try:
            df_stage_1, df_stage_2, df_stage_3, df_stage_4, df_stage_5 = self.read_df()
            df_stage_1 = self.preprocess_kpi_df(df_stage_1, kpi="Quality", process_stage_name="Stage-01",
                                                process_stage_id="P2E-STAGE-001")
            df_stage_2 = self.preprocess_kpi_df(df_stage_2, kpi="Quality", process_stage_name="Stage-02",
                                                process_stage_id="P2E-STAGE-002")
            df_stage_3 = self.preprocess_kpi_df(df_stage_3, kpi="Quality", process_stage_name="Stage-03",
                                                process_stage_id="P2E-STAGE-003")
            df_stage_4 = self.preprocess_kpi_df(df_stage_4, kpi="Quality", process_stage_name="Stage-04",
                                                process_stage_id="P2E-STAGE-004")
            df_stage_5 = self.preprocess_kpi_df(df_stage_5, kpi="Quality", process_stage_name="Stage-05",
                                                process_stage_id="P2E-STAGE-005")

            df = pd.concat([df_stage_1, df_stage_2, df_stage_3, df_stage_4, df_stage_5], axis=0)
            # print(df.columns)
            df = df[['batch_no', 'process_stage_name', 'process_stage_id',
                     'batch_start_time', 'batch_end_time', 'kpi_name',
                     'kpi_description', 'kpi_value']]
            df.reset_index(drop=True, inplace=True)
            df.to_excel(f"{base_path}/batch_kpi_master.xlsx", index=False)
            logger.info(f'Pushing batch_kpi_master to postgres')
            df.set_index('batch_no').to_sql("batch_kpi_master",
                                            "postgresql://ilens:iLens$456@192.168.0.207:5328/ilens_ai",
                                            if_exists="replace")
            logger.debug(f'Pushed batch_kpi_master to postgres')
            return df
        except Exception as e:
            logger.exception(f"Exception - {e}")

    def orchestrator_master(self):
        try:
            df_stage_1, df_stage_2, df_stage_3, df_stage_4, df_stage_5 = self.read_df()
            df_stage_1 = self.preprocess_master_df(df_stage_1, process_stage_name="Stage-01",
                                                   process_stage_id="P2E-STAGE-001",
                                                   batch_product="P2E-Stage-01", ideal_batch_cycle_time_hr=12,
                                                   batch_setup_time_hr=1)
            df_stage_2 = self.preprocess_master_df(df_stage_2, process_stage_name="Stage-02",
                                                   process_stage_id="P2E-STAGE-002",
                                                   batch_product="P2E-Stage-02", ideal_batch_cycle_time_hr=24,
                                                   batch_setup_time_hr=1)
            df_stage_3 = self.preprocess_master_df(df_stage_3, process_stage_name="Stage-03",
                                                   process_stage_id="P2E-STAGE-003",
                                                   batch_product="P2E-Stage-03", ideal_batch_cycle_time_hr=48,
                                                   batch_setup_time_hr=1)
            df_stage_4 = self.preprocess_master_df(df_stage_4, process_stage_name="Stage-04",
                                                   process_stage_id="P2E-STAGE-004",
                                                   batch_product="P2E-Stage-04", ideal_batch_cycle_time_hr=124,
                                                   batch_setup_time_hr=1)
            df_stage_5 = self.preprocess_master_df(df_stage_5, process_stage_name="Stage-05",
                                                   process_stage_id="P2E-STAGE-005",
                                                   batch_product="P2E-Stage-05", ideal_batch_cycle_time_hr=None,
                                                   batch_setup_time_hr=1)

            df = pd.concat([df_stage_1, df_stage_2, df_stage_3, df_stage_4, df_stage_5], axis=0)
            df.reset_index(drop=True, inplace=True)
            df.to_excel(f"{base_path}/batch_master.xlsx", index=False)
            logger.info(f'Pushing batch_master to postgres')
            df.set_index('batch_no').to_sql("batch_master",
                                            "postgresql://ilens:iLens$456@192.168.0.207:5328/ilens_ai",
                                            if_exists="replace")

            logger.debug(f'Pushed batch_master to postgres')
            return df
        except Exception as e:
            logger.exception(f"Exception - {e}")

    @staticmethod
    def join_df_proto(df_master, df_other):
        try:
            df = pd.merge(left=df_master, right=df_other, how='left', on='batch_no')
            df['bct_input_qty_ratio'] = df['batch_cycle_time_minutes']/df['input_qty']
            df.reset_index(drop=True, inplace=True)
            df.to_excel(f"{base_path}/master_proto.xlsx", index=False)
            logger.info(f'Pushing master_join to postgres')
            df.set_index('batch_no').to_sql("master_proto", "postgresql://ilens:iLens$456@192.168.0.207:5328/ilens_ai", if_exists='replace')
            logger.debug(f'Pushed master_join to postgres')
        except Exception as e:
            logger.exception(f"Exception - {e}")

    def orchestrator_master_proto(self):
        try:
            df_stage_1, df_stage_2, df_stage_3, df_stage_4, df_stage_5 = self.read_df()
            df_stage_1 = self.preprocess_master_df_proto(df_stage_1, process_stage_name="Stage-01")
            df_stage_2 = self.preprocess_master_df_proto(df_stage_2, process_stage_name="Stage-02")
            df_stage_3 = self.preprocess_master_df_proto(df_stage_3, process_stage_name="Stage-03")
            df_stage_4 = self.preprocess_master_df_proto(df_stage_4, process_stage_name="Stage-04")
            df_stage_5 = self.preprocess_master_df_proto(df_stage_5, process_stage_name="Stage-05")

            df = pd.concat([df_stage_1, df_stage_2, df_stage_3, df_stage_4, df_stage_5], axis=0)
            df.reset_index(drop=True, inplace=True)
            df.to_excel(f"{base_path}/batch_master_quantity.xlsx", index=False)
            logger.info(f'Pushing batch_master to postgres')
            df.set_index('batch_no').to_sql("batch_master_quantity",
                                            "postgresql://ilens:iLens$456@192.168.0.207:5328/ilens_ai",
                                            if_exists="replace")
            logger.debug(f'Pushed batch_master to postgres')
            return df
        except Exception as e:
            logger.exception(f"Exception - {e}")


if __name__=="__main__":
    batch_master = BatchMaster()
    df_master = batch_master.orchestrator_master()
    # df_kpi = batch_master.orchestrator_kpi()
    df_master_quantity = batch_master.orchestrator_master_proto()
    # batch_master.join_df(df_master=df_master, df_kpi=df_kpi)
    batch_master.join_df_proto(df_master=df_master, df_other=df_master_quantity)
    logger.info(f'{df_master.shape}')
