import pandas as pd
from loguru import logger
import warnings
warnings.filterwarnings('ignore')

base_path = f'data'


class BatchMaster:
    def __init__(self):
        self.sheet = "P2E Golden batch 05-01-2023.xlsx"

    def read_file(self, sheet_name, skiprows=1):
        try:
            df = pd.read_excel(f'{base_path}/{self.sheet}', sheet_name=sheet_name,
                               skiprows=skiprows)
            return df
        except Exception as e:
            logger.exception(f"Exception - {e}")

    def read_df(self):
        try:
            df_stage_1 = self.read_file(sheet_name="Stage I")
            df_stage_2 = self.read_file(sheet_name="Stage II")
            df_stage_3 = self.read_file(sheet_name="Stage III")
            df_stage_4 = self.read_file(sheet_name="Stage IV")
            df_stage_5 = self.read_file(sheet_name="Stage V")
            logger.debug(f'{df_stage_1.shape}, {df_stage_2.shape}, {df_stage_3.shape}, {df_stage_4.shape},'
                         f'{df_stage_5.shape},')

            return df_stage_1, df_stage_2, df_stage_3, df_stage_4, df_stage_5
        except Exception as e:
            logger.exception(f"Exception - {e}")

    @staticmethod
    def batch_master(df, process_stage_name, batch_product, ideal_batch_cycle_time_hr, batch_setup_time_hr,
                             process_stage_id, work_order_no="P2E-STD-01", work_order_item_no="P2E-STD-01-01",
                             final_product="P2E"):
        try:

            df = df[['Batch  Number', 'Start Time.1', 'End Time.1', 'Downtime', 'Equipment']]
            df.rename(columns={'Batch  Number': 'batch_no', 'Start Time.1': 'batch_start_time',
                               'End Time.1': 'batch_end_time', 'Downtime': 'downtime',
                               'Equipment': 'used_equipment'}, inplace=True)
            df['live_batch_end_time'] = None
            df['process_stage_name'] = process_stage_name
            df['process_stage_id'] = process_stage_id
            df['work_order_no'] = work_order_no
            df['work_order_item_no'] = work_order_item_no
            df['final_product'] = final_product
            df['batch_product'] = batch_product
            df['ideal_batch_cycle_time_hr'] = ideal_batch_cycle_time_hr
            df['batch_cycle_time'] = df['batch_end_time'] - df['batch_start_time']
            df['batch_cycle_time_minutes'] = df['batch_cycle_time'].dt.total_seconds() // 60
            df['batch_setup_time_hr'] = batch_setup_time_hr
            df['selected_equipments'] = df['used_equipment']
            df['created_on'] = df['batch_start_time'].dt.date
            df['created_by'] = 'Aakash'
            df['last_updated_on'] = df['batch_end_time'].dt.date
            df['last_updated_by'] = 'Aakash'
            df['golden_batch'] = None
            df = df.astype({'batch_cycle_time': str})

            df = df[['batch_no', 'final_product', 'batch_product', 'process_stage_name', 'process_stage_id',
                     'work_order_no', 'work_order_item_no', 'ideal_batch_cycle_time_hr', 'batch_start_time',
                     'batch_end_time', 'live_batch_end_time', 'batch_cycle_time', 'batch_cycle_time_minutes',
                     'batch_setup_time_hr', 'downtime', 'selected_equipments', 'created_on', 'created_by',
                     'last_updated_on', 'last_updated_by', 'golden_batch']]
            df.reset_index(drop=True, inplace=True)
            return df
        except Exception as e:
            logger.exception(f"Exception - {e}")

    @staticmethod
    def batch_material_master(df, process_stage_name):
        try:
            df.rename(columns={'Batch  Number': 'batch_no', 'Start Time.1': 'batch_start_time',
                               'End Time.1': 'batch_end_time', 'Downtime': 'downtime',
                               'Equipment': 'used_equipment'}, inplace=True)
            df_stage = pd.DataFrame(index=[i for i in range(len(df))])
            if process_stage_name == "Stage-01":
                df_stage['batch_no'] = df['batch_no']
                df_stage['input_qty'] = df['Input Qty. (kg) F00056-BULK-001']
                df_stage['input_qty_uom'] = 'kg'
                df_stage['output_qty'] = df['Output Qty. (kg)']
                df_stage['output_qty_uom'] = 'kg'

            elif process_stage_name == "Stage-02":
                df_stage['batch_no'] = df['batch_no']
                df_stage['input_qty'] = df['Input Qty.\n(kg) Stage-I']
                df_stage['input_qty_uom'] = 'kg'
                df_stage['output_qty'] = df['Output \nQty. (kg)']
                df_stage['output_qty_uom'] = 'kg'

            elif process_stage_name == "Stage-03":
                df_stage['batch_no'] = df['batch_no']
                df_stage['input_qty'] = df['Input Qty. (kg) Stage-I']
                df_stage['input_qty_uom'] = 'kg'
                df_stage['output_qty'] = df['Output Qty. (kg)']
                df_stage['output_qty_uom'] = 'kg'

            elif process_stage_name == "Stage-04":
                df_stage['batch_no'] = df['batch_no']
                df_stage['input_qty'] = df['Input Qty. (kg)\n/Stage-I']
                df_stage['input_qty_uom'] = 'kg'
                df_stage['output_qty'] = df['Output \nQty. (kg)']
                df_stage['output_qty_uom'] = 'kg'

            elif process_stage_name == "Stage-05":
                df_stage['batch_no'] = df['batch_no']
                df_stage['input_qty'] = df['Input \nQty. Kg\nStage-I']
                df_stage['input_qty_uom'] = 'kg'
                df_stage['output_qty'] = df['Output \nQty. (kg)']
                df_stage['output_qty_uom'] = 'kg'

            df_stage.reset_index(drop=True, inplace=True)
            df_stage = df_stage[['batch_no', 'input_qty', 'input_qty_uom', 'output_qty', 'output_qty_uom']]

            return df_stage
        except Exception as e:
            logger.exception(f"Exception - {e}")

    @staticmethod
    def batch_kpi_master(df, process_stage_name, process_stage_id):
        try:
            df.rename(columns={'Batch  Number': 'batch_no', 'Start Time.1': 'batch_start_time',
                               'End Time.1': 'batch_end_time', 'Quality': 'quality'}, inplace=True)
            df['process_stage_name'] = process_stage_name
            df['process_stage_id'] = process_stage_id

            if process_stage_name == "Stage-01":
                df['formaldehyde_content'] = df['F00041-BULK-001 content nil']
                df['unreactive_alpha_picoline'] = df['un reacted F00056-BULK-001 ( for infor.)']
                df['moisture_content'] = df['Moisture content \n(for infor)']

            elif process_stage_name == "Stage-02":
                df['formaldehyde_content'] = None
                df['unreactive_alpha_picoline'] = df['F00056-BULK-001 \n(NLT 50 %)']
                df['moisture_content'] = df['Moisture content \n(for Information).1']

            elif process_stage_name == "Stage-03":
                df['formaldehyde_content'] = None
                df['unreactive_alpha_picoline'] = None
                df['moisture_content'] = df['M/C for infor']

            elif process_stage_name == "Stage-04":
                df['formaldehyde_content'] = None
                df['unreactive_alpha_picoline'] = None
                df['moisture_content'] = df['NMT 0.5%']

            elif process_stage_name == "Stage-05":
                df['formaldehyde_content'] = None
                df['unreactive_alpha_picoline'] = None
                df['moisture_content'] = df[' NMT 1.0%']

            df = df[['batch_no', 'quality', 'formaldehyde_content', 'unreactive_alpha_picoline',
                     'moisture_content']]
            return df
        except Exception as e:
            logger.exception(f"Exception - {e}")

    def orchestrator_master(self):
        try:
            df_stage_1, df_stage_2, df_stage_3, df_stage_4, df_stage_5 = self.read_df()
            df_stage_1 = self.batch_master(df_stage_1, process_stage_name="Stage-01",
                                           process_stage_id="P2E-STAGE-001",
                                           batch_product="P2E-Stage-01", ideal_batch_cycle_time_hr=720,
                                           batch_setup_time_hr=1)
            df_stage_2 = self.batch_master(df_stage_2, process_stage_name="Stage-02",
                                           process_stage_id="P2E-STAGE-002",
                                           batch_product="P2E-Stage-02", ideal_batch_cycle_time_hr=1440,
                                           batch_setup_time_hr=1)
            df_stage_3 = self.batch_master(df_stage_3, process_stage_name="Stage-03",
                                           process_stage_id="P2E-STAGE-003",
                                           batch_product="P2E-Stage-03", ideal_batch_cycle_time_hr=2880,
                                           batch_setup_time_hr=1)
            df_stage_4 = self.batch_master(df_stage_4, process_stage_name="Stage-04",
                                           process_stage_id="P2E-STAGE-004",
                                           batch_product="P2E-Stage-04", ideal_batch_cycle_time_hr=7440,
                                           batch_setup_time_hr=1)
            df_stage_5 = self.batch_master(df_stage_5, process_stage_name="Stage-05",
                                           process_stage_id="P2E-STAGE-005",
                                           batch_product="P2E-Stage-05", ideal_batch_cycle_time_hr=6570,
                                           batch_setup_time_hr=1)

            df = pd.concat([df_stage_1, df_stage_2, df_stage_3, df_stage_4, df_stage_5], axis=0)
            df.reset_index(drop=True, inplace=True)

            df = df.astype({'ideal_batch_cycle_time_hr': float})
            df = df.round(2)
            df.to_excel(f"{base_path}/t_batch_master.xlsx", index=False)
            logger.info(f'Pushing batch_master to postgres')
            df.set_index('batch_no').to_sql("t_batch_master",
                                            "postgresql://ilens:iLens$456@192.168.0.207:5455/ilens_ai",
                                            if_exists="replace")

            logger.debug(f'Pushed batch_master to postgres')
            return df
        except Exception as e:
            logger.exception(f"Exception - {e}")

    def orchestrator_batch_material_master(self):
        try:
            df_stage_1, df_stage_2, df_stage_3, df_stage_4, df_stage_5 = self.read_df()
            df_stage_1 = self.batch_material_master(df_stage_1, process_stage_name="Stage-01")
            df_stage_2 = self.batch_material_master(df_stage_2, process_stage_name="Stage-02")
            df_stage_3 = self.batch_material_master(df_stage_3, process_stage_name="Stage-03")
            df_stage_4 = self.batch_material_master(df_stage_4, process_stage_name="Stage-04")
            df_stage_5 = self.batch_material_master(df_stage_5, process_stage_name="Stage-05")

            df = pd.concat([df_stage_1, df_stage_2, df_stage_3, df_stage_4, df_stage_5], axis=0)
            df.reset_index(drop=True, inplace=True)
            df = df[['batch_no', 'input_qty', 'input_qty_uom', 'output_qty', 'output_qty_uom']]
            df = df.round(2)
            df.to_excel(f"{base_path}/t_batch_material_master.xlsx", index=False)
            logger.info(f'Pushing batch_master to postgres')
            df.set_index('batch_no').to_sql("t_batch_material_master",
                                            "postgresql://ilens:iLens$456@192.168.0.207:5455/ilens_ai",
                                            if_exists="replace")
            logger.debug(f'Pushed batch_master to postgres')
            return df
        except Exception as e:
            logger.exception(f"Exception - {e}")

    def orchestrator_batch_kpi_master(self):
        try:
            df_stage_1, df_stage_2, df_stage_3, df_stage_4, df_stage_5 = self.read_df()
            df_stage_1 = self.batch_kpi_master(df_stage_1, process_stage_name="Stage-01",
                                               process_stage_id="P2E-STAGE-001")
            df_stage_2 = self.batch_kpi_master(df_stage_2, process_stage_name="Stage-02",
                                               process_stage_id="P2E-STAGE-002")
            df_stage_3 = self.batch_kpi_master(df_stage_3, process_stage_name="Stage-03",
                                               process_stage_id="P2E-STAGE-003")
            df_stage_4 = self.batch_kpi_master(df_stage_4, process_stage_name="Stage-04",
                                               process_stage_id="P2E-STAGE-004")
            df_stage_5 = self.batch_kpi_master(df_stage_5, process_stage_name="Stage-05",
                                               process_stage_id="P2E-STAGE-005")

            df = pd.concat([df_stage_1, df_stage_2, df_stage_3, df_stage_4, df_stage_5], axis=0)
            df = df[['batch_no', 'quality', 'formaldehyde_content', 'unreactive_alpha_picoline', 'moisture_content']]
            df.reset_index(drop=True, inplace=True)
            df = df.round(2)
            df.to_excel(f"{base_path}/t_batch_kpi_master.xlsx", index=False)
            logger.info(f'Pushing batch_kpi_master to postgres')
            df.set_index('batch_no').to_sql("t_batch_kpi_master",
                                            "postgresql://ilens:iLens$456@192.168.0.207:5455/ilens_ai",
                                            if_exists="replace")
            logger.debug(f'Pushed batch_kpi_master to postgres')
            return df
        except Exception as e:
            logger.exception(f"Exception - {e}")

    @staticmethod
    def join_master(df_batch_master, df_batch_kpi_master, df_batch_material_master):
        try:
            df_kpi_join = pd.merge(left=df_batch_master, right=df_batch_kpi_master, how='left', on='batch_no')
            df = pd.merge(left=df_kpi_join, right=df_batch_material_master, how='left', on='batch_no')
            df['bct_input_qty_ratio'] = df['batch_cycle_time_minutes']/df['input_qty']

            min_bct_1 = df.loc[df['process_stage_id'] == 'P2E-STAGE-001', 'batch_cycle_time_minutes'].min()
            min_bct_2 = df.loc[df['process_stage_id'] == 'P2E-STAGE-002', 'batch_cycle_time_minutes'].min()
            min_bct_3 = df.loc[df['process_stage_id'] == 'P2E-STAGE-003', 'batch_cycle_time_minutes'].min()
            min_bct_4 = df.loc[df['process_stage_id'] == 'P2E-STAGE-004', 'batch_cycle_time_minutes'].min()
            min_bct_5 = df.loc[df['process_stage_id'] == 'P2E-STAGE-005', 'batch_cycle_time_minutes'].min()
            df.loc[(df['process_stage_id'] == 'P2E-STAGE-001') & (df['batch_cycle_time_minutes'] == min_bct_1),
                   'golden_batch'] = 'Golden Batch'
            df.loc[(df['process_stage_id'] == 'P2E-STAGE-002') & (df['batch_cycle_time_minutes'] == min_bct_2),
                   'golden_batch'] = 'Golden Batch'
            df.loc[(df['process_stage_id'] == 'P2E-STAGE-003') & (df['batch_cycle_time_minutes'] == min_bct_3),
                   'golden_batch'] = 'Golden Batch'
            df.loc[(df['process_stage_id'] == 'P2E-STAGE-004') & (df['batch_cycle_time_minutes'] == min_bct_4),
                   'golden_batch'] = 'Golden Batch'
            df.loc[(df['process_stage_id'] == 'P2E-STAGE-005') & (df['batch_cycle_time_minutes'] == min_bct_5),
                   'golden_batch'] = 'Golden Batch'
            df.reset_index(drop=True, inplace=True)
            df = df.round(2)
            df.to_excel(f"{base_path}/t_batch_information_master.xlsx", index=False)
            logger.info(f'Pushing master_join to postgres')
            df.set_index('batch_no').to_sql("t_batch_information_master",
                                            "postgresql://ilens:iLens$456@192.168.0.207:5455/ilens_ai", if_exists='replace')
            logger.debug(f'Pushed master_join to postgres')
        except Exception as e:
            logger.exception(f"Exception - {e}")


if __name__=="__main__":
    batch_master = BatchMaster()
    df_batch_master = batch_master.orchestrator_master()
    df_batch_kpi_master = batch_master.orchestrator_batch_kpi_master()
    df_batch_material_master = batch_master.orchestrator_batch_material_master()
    batch_master.join_master(df_batch_master=df_batch_master,
                             df_batch_kpi_master=df_batch_kpi_master,
                             df_batch_material_master=df_batch_material_master)
    logger.info('Module Completed')
