import base64
import os
import time
from copy import deepcopy
from datetime import datetime

from scripts.config import app_configuration
from scripts.config.app_constants import DBMapping, CaseStatus, StatusMessages, FILE_PATH, StaticJsons
from scripts.config.db_connection_obj import ConnectionObj
from scripts.logging.logger import logger
from scripts.utils.get_new_id import GetNewId
from scripts.utils.mongo_utility import MongoConnect


class TicketSetupHandler:
    def __init__(self):
        try:
            logger.debug("Inside the Resolver setup module")
            self.new_id = GetNewId()
            self.mongo_obj = ConnectionObj.mongo_connection_obj
            if not ConnectionObj.mongo_connection_obj:
                self.mongo_obj = ConnectionObj.mongo_connection_obj = MongoConnect()
        except Exception as e:
            logger.exception("Exception in the data utility definition" + str(e))

    def save_ticket_data(self, input_json):
        final_json = {"status": StatusMessages.FAILED, "message": StatusMessages.FAILED}
        try:
            logger.debug("Inside save ticket data definition")
            category_data = list(self.mongo_obj.aggregate(
                db_name=DBMapping.support_lens_configuration,
                collection_name=DBMapping.category_configuration,
                list_for_aggregation=[
                    {
                        '$match': {
                            'category_id': input_json["category"]
                        }
                    }, {
                        '$project': {
                            '_id': 0,
                            "CategoryName": "$CategoryName"
                        }
                    }
                ]))
            project_data = self.mongo_obj.find_one(db_name=DBMapping.ilens_configuration,
                                                   collection_name=DBMapping.customer_projects,
                                                   query={"customer_project_id": input_json["project_id"]})
            subcategory_data = list(self.mongo_obj.aggregate(
                db_name=DBMapping.support_lens_configuration,
                collection_name=DBMapping.subcategory_configuration,
                list_for_aggregation=[
                    {
                        '$match': {
                            'sub_category_id': input_json["subCategory"]
                        }
                    }, {
                        '$project': {
                            '_id': 0,
                            "subCategoryName": "$subCategoryName"
                        }
                    }
                ]))
            site_data = list(self.mongo_obj.aggregate(
                db_name=DBMapping.ilens_configuration,
                collection_name=DBMapping.site_conf,
                list_for_aggregation=[
                    {
                        '$match': {
                            'site_id': input_json["site"]
                        }
                    }, {
                        '$project': {
                            '_id': 0
                        }
                    }
                ]))
            # user_data = list(self.mongo_obj.aggregate(
            #     db_name=DBMapping.ilens_configuration,
            #     collection_name=DBMapping.user,
            #     list_for_aggregation=[
            #         {
            #             '$match': {
            #                 'user_id': {"$in": input_json["contacts"]}
            #             }
            #         }, {
            #             '$project': {
            #                 '_id': 0
            #             }
            #         }
            #     ]))
            if len(category_data):
                category_data = category_data[0]
            else:
                final_json = dict(message="Invalid category selected", status=StatusMessages.FAILED)
                return final_json
            if len(subcategory_data):
                subcategory_data = subcategory_data[0]
            else:
                final_json = dict(message="Invalid subcategory selected", status=StatusMessages.FAILED)
                return final_json
            if len(site_data):
                site_data = site_data[0]
            else:
                final_json = dict(message="Invalid Site selected", status=StatusMessages.FAILED)
                return final_json
            user = self.new_id.get_user_id()
            if "user_id" in user:
                user_data = self.mongo_obj.find_one(db_name=DBMapping.ilens_configuration,
                                                    collection_name=DBMapping.user,
                                                    query={"user_id": user["user_id"]})
                user = dict(userName=user_data["username"], user_id=user.get("user_id"))
            if input_json["type"] == "new":
                case_id = "case_" + self.new_id.get_next_id("case")
            else:
                case_id = input_json["case_id"]
            if "deletedList" in input_json and len(input_json["deletedList"]):
                for file in input_json["deletedList"]:
                    response = self.delete_file_details(file, input_json["case_id"])
            if len(input_json["fileNameBlock"]):
                for file_data in input_json["fileNameBlock"]:
                    if "attachment" not in file_data:
                        continue
                    response = self.save_file_details(file_data, case_id)
                    del file_data["attachment"]
            if input_json["type"].lower() == "new":
                insert_json = dict(CategoryName=category_data["CategoryName"],
                                   ProjectName=project_data["customer_project_name"],
                                   category_id=input_json["category"],
                                   subCategoryName=subcategory_data["subCategoryName"],
                                   sub_category_id=input_json["subCategory"],
                                   Description=input_json["description"],
                                   subject=input_json["subject"],
                                   site_id=input_json["site"],
                                   siteName=site_data["site_name"],
                                   case_id=case_id,
                                   project_id=input_json["project_id"],
                                   case_status=CaseStatus.OPEN,
                                   created_on=time.time(),
                                   created_by=user,
                                   file_details=input_json["fileNameBlock"],
                                   last_updated_on=time.time(),
                                   last_updated_by=user,
                                   user_details=input_json["contacts"]
                                   )
                self.mongo_obj.insert_one(database_name=DBMapping.support_lens_configuration,
                                          collection_name=DBMapping.ticket_configuration,
                                          json_data=insert_json)
            if input_json["type"].lower() == "edit":
                query_json = {"case_id": input_json["case_id"]}
                case_data = self.mongo_obj.find_one(db_name=DBMapping.support_lens_configuration,
                                                    collection_name=DBMapping.ticket_configuration,
                                                    query=query_json,
                                                    search_json={"_id": 0})
                if len(case_data):
                    case_data.update(CategoryName=category_data["CategoryName"],
                                     category_id=input_json["category"],
                                     subCategoryName=subcategory_data["subCategoryName"],
                                     sub_category_id=input_json["subCategory"],
                                     Description=input_json["description"],
                                     subject=input_json["subject"],
                                     site_id=input_json["site"],
                                     siteName=site_data["site_name"],
                                     project_id=input_json["project_id"],
                                     case_status=CaseStatus.OPEN,
                                     file_details=input_json["fileNameBlock"],
                                     last_updated_on=time.time(),
                                     last_updated_by=user,
                                     user_details=input_json["contacts"]
                                     )
                    if "model" in input_json:
                        if input_json["model"].lower() == "view":
                            case_data.update(comments=input_json.get("comments"),
                                             assigned_to=input_json.get("assign"),
                                             case_status=input_json.get("status"),
                                             emailIds=input_json.get("email"),
                                             phoneNumbers=input_json.get("phoneNumber"))
                    self.mongo_obj.update_one(db_name=DBMapping.support_lens_configuration,
                                              collection_name=DBMapping.ticket_configuration,
                                              set_json=case_data, query=query_json)
            final_json = dict(status=StatusMessages.SUCCESS, message="Case data saved successfully")
        except Exception as e:
            logger.exception("Exception while saving ticket data" + str(e))
        return final_json

    def save_file_details(self, input_json, case_id):
        try:
            file_save_path = self.check_file_extenstions(input_json["name"], case_id)
            if not file_save_path:
                return False
            decoded_data = base64.b64decode(str(input_json["attachment"]))
            if not os.path.exists(file_save_path):
                os.makedirs(file_save_path)
            with open(f"{file_save_path}/{input_json['name']}", "wb") as file:
                file.write(decoded_data)
                file.close()
            return True
        except Exception as e:
            logger.exception("Exception while saving ticket data" + str(e))
            return False

    def delete_file_details(self, file_name, case_id):
        try:
            file_delete_path = self.check_file_extenstions(file_name, case_id)
            if not file_delete_path:
                return False
            if os.path.exists(f"{file_delete_path}\{file_name}"):
                os.remove(f"{file_delete_path}\{file_name}")
            return True
        except Exception as e:
            logger.exception("Exception while saving ticket data" + str(e))
            return False

    def check_file_extenstions(self, file_name, case_id):
        file_extenstion = file_name.split(".")[-1].lower()
        if file_extenstion in ["csv"]:
            file_save_path = os.path.join(app_configuration.FILES_SAVE_PATH, FILE_PATH.CSV_PATH, case_id)
        elif file_extenstion in ["jpg", "jpeg", "png", "svg"]:
            file_save_path = os.path.join(app_configuration.FILES_SAVE_PATH, FILE_PATH.IMAGES, case_id)
        elif file_extenstion in ["txt"]:
            file_save_path = os.path.join(app_configuration.FILES_SAVE_PATH, FILE_PATH.TXT, case_id)
        elif file_extenstion in ["pdf"]:
            file_save_path = os.path.join(app_configuration.FILES_SAVE_PATH, FILE_PATH.PDF, case_id)
        elif file_extenstion in ["docs", "docx", "doc"]:
            file_save_path = os.path.join(app_configuration.FILES_SAVE_PATH, FILE_PATH.DOCS, case_id)
        else:
            return False
        return file_save_path

    def fetch_ticket_details(self, input_json):
        final_json = dict(status=StatusMessages.FAILED, message=StatusMessages.FAILED, data=list())
        try:
            query_json = {
                '_id': 0,
                'category': "$category_id",
                'subCategory': '$sub_category_id',
                'site': '$site_id',
                'subject': "$subject",
                'fileNameBlock': '$file_details',
                'description': '$Description',
                'tab_type': '$case_status',
                'contacts': '$user_details'
            }
            if "model" in input_json:
                if input_json["model"].lower() == "view":
                    query_json.update({
                        "status": "$case_status",
                        "assign": "$assigned_to",
                        "comments": "$comments",
                        "phoneNumber": "$phoneNumbers",
                        "email": "$emailIds"
                    })
            response_data = list(self.mongo_obj.aggregate(
                db_name=DBMapping.support_lens_configuration,
                collection_name=DBMapping.ticket_configuration,
                list_for_aggregation=[
                    {
                        '$match': {
                            'case_id': input_json["case_id"]
                        }
                    }, {
                        '$project': query_json
                    }
                ]))

            if len(response_data):
                response_data = response_data[0]
            final_json = dict(status=StatusMessages.SUCCESS, message=StatusMessages.SUCCESS,
                              data=deepcopy(response_data))

        except Exception as e:
            logger.exception("Exception while fetching case details" + str(e))
        return final_json

    def delete_support_case_details(self, input_json):
        try:
            response = self.update_case_status(case_id=input_json["case_id"], status_value=CaseStatus.DELETED)
            return {"status": "success", "message": "Case deleted successfully"}
        except Exception as e:
            logger.exception(str(e))
            return {"status": "failed", "message": str(e)}

    def resolve_support_case_details(self, input_json):
        try:
            response = self.update_case_status(case_id=input_json["case_id"], status_value=CaseStatus.RESOLVED)
            return {"status": "success", "message": "Case resolved successfully"}
        except Exception as e:
            logger.exception(str(e))
            return {"status": "failed", "message": str(e)}

    def close_case_details(self, input_json):
        try:
            response = self.update_case_status(case_id=input_json["case_id"], status_value=CaseStatus.CLOSED)
            return {"status": "success", "message": "Case closed successfully"}
        except Exception as e:
            logger.exception(str(e))
            return {"status": "failed", "message": str(e)}

    @staticmethod
    def update_case_status(case_id, status_value):
        try:
            query = {"case_id": case_id}
            new_values = {"$set": {"case_status": status_value, "last_updated_on": time.time()}}
            ConnectionObj.mongo_connection_obj.update_one(db_name=DBMapping.support_lens_configuration,
                                                          collection_name=DBMapping.ticket_configuration,
                                                          query=query, set_json=new_values)
            return True
        except Exception as e:
            logger.error(f"Exception occurred while updating case status {str(e)}", exc_info=True)
            return False

    def get_support_case_table_details(self, input_json):
        final_json = dict(status=StatusMessages.FAILED, message=StatusMessages.FAILED,
                          data=dict(tableData=dict(headerContent=StaticJsons.SUPPORTLENS_FETCHTABLE_HEADERCONTENT),
                                    tableActions=StaticJsons.SUPPORTLENS_FETCHTABLE_TABLEACTIONS,
                                    enableRowExpand=True, table_type="infinite_scroll", hideSearch=True,
                                    server_search=True))

        try:
            json_data = [
                {
                    '$match': {
                        'project_id': input_json["project_id"],
                        'case_status': {"$in": input_json["tab_name"]}
                    }
                }, {
                    '$project': {
                        '_id': 0,
                        'case_id': '$case_id',
                        'status': '$case_status',
                        "subject": '$subject',
                        "ProjectName": "$ProjectName",
                        'last_updated_on': '$last_updated_on',
                        'created_by': '$created_by'
                    }
                },
                {
                    '$sort': {
                        'last_updated_on': -1
                    }
                },
                {
                    '$limit': input_json["records"]
                },
                {
                    '$skip': int(input_json["records"]) * (int(input_json["page"]) - 1)
                }
            ]
            if input_json["filters"]:
                for key, value in input_json["filters"].items():
                    if value not in ["", None, str()]:
                        if key == "created_by":
                            json_data[0]["$match"].update(
                                {f"{key}.userName": {"$regex": f"{value}", '$options': "i"}})
                        elif key == "last_updated_on":
                            json_data[0]["$match"].update(
                                {key: {"$gte": value[0] / 1000, '$lte': value[-1] / 1000}})
                        else:
                            json_data[0]["$match"].update(
                                {key: {"$regex": f"{value}", '$options': "i"}})
            records = list(self.mongo_obj.aggregate(
                db_name=DBMapping.support_lens_configuration,
                collection_name=DBMapping.ticket_configuration,
                list_for_aggregation=json_data))
            json_data.pop(3)
            json_data.pop(2)
            total_records = list(self.mongo_obj.aggregate(
                db_name=DBMapping.support_lens_configuration,
                collection_name=DBMapping.ticket_configuration,
                list_for_aggregation=json_data))
            users_list = list()
            for record in records:
                print()
                time_diff = datetime.fromtimestamp(time.time()) - datetime.fromtimestamp(record["last_updated_on"])
                days_obj = time_diff.days
                hour_obj = time_diff.seconds // 3600
                minutes_obj = time_diff.seconds // 60 % 60

                record["created_by"] = record["created_by"]["userName"].replace("_", " ").replace("-", " ").title()
                record["last_updated_on"] = datetime.fromtimestamp(record["last_updated_on"]).strftime("%d %b %Y,%H:%M")
                record["expandData"] = [{"label": str(key).replace("_", " ").replace("-", " ").title(), "value": value}
                                        for key, value in record.items()]
                if record["created_by"] not in users_list:
                    users_list.append(record["created_by"])
                if record["status"].lower() == "open":
                    record["status"] = record["status"].title()
                    record["svgName"] = "open.svg"
                if days_obj != 0:
                    record["time"] = f"{days_obj} days {hour_obj} hour {minutes_obj} minutes ago"
                else:
                    if hour_obj != 0:
                        record["time"] = f"{hour_obj} hour {minutes_obj} minutes ago"
                    else:
                        record["time"] = f"{minutes_obj} minutes ago"

            users_data = deepcopy(final_json["data"]["tableData"]["headerContent"][-2])
            users_data["options"] = [{"label": str(user).replace("_", " ").replace("-", " ").title(), "value": user}
                                     for user in users_list]
            if len(input_json["tab_name"]):
                if input_json["tab_name"][0] in ["closed","deleted"]:
                    final_json["data"]["tableActions"] = dict()

            final_json["data"]["tableData"]["headerContent"][-2] = deepcopy(users_data)
            final_json.update(status=StatusMessages.SUCCESS, message=StatusMessages.SUCCESS)
            final_json["data"]["tableData"].update(bodyContent=deepcopy(records))
            final_json["data"]["total_no"] = len(total_records)
            if len(total_records) <= (int(input_json["records"]) * int(input_json["page"])):
                final_json["data"]["end_of_records"] = True
            else:
                final_json["data"]["end_of_records"] = False
            if input_json["enable_tab"]:
                final_json["tab_count"] = self.get_case_count_details(input_json=input_json,
                                                                      filter_value="support_lens_table_header_content")
        except Exception as e:
            logger.exception(str(e))
        return final_json

    def get_dashboard_details(self, input_json):
        final_json = dict(status=StatusMessages.FAILED, message=StatusMessages.FAILED, data=dict())
        try:
            final_json["data"]["caseStatus"] = self.get_case_count_details(input_json=input_json,
                                                                           filter_value="support_lens_dashboard_header_content")
            final_json["data"]["chartData"] = self.get_chart_details(input_json)
            final_json.update(status="success", message="success")

        except Exception as e:
            logger.exception(str(e))
        return final_json

    def get_chart_details(self, input_json):
        try:
            chart_json = self.get_case_count_details(input_json=input_json,
                                                     filter_value="support_lens_chart_header_content")
            total_count = 0
            for chart in chart_json:
                chart["value"] = chart.pop("count")
                total_count += chart["value"]

            for data in chart_json:
                data["percentage"] = f"{round(float((data['value'] / total_count) * 100), 2)}%"
            return chart_json

        except Exception as e:
            logger.exception(str(e))
        return dict()

    def get_case_count_details(self, input_json, filter_value):
        final_json = dict(status=StatusMessages.SUCCESS, message=StatusMessages.SUCCESS, data=dict())
        count_json_header_content = deepcopy(StaticJsons.SUPPORTLENS_HEADER_CONTENT[filter_value])
        try:
            json_data = [
                {
                    '$match': {
                        'project_id': input_json["project_id"]
                    }
                },
                {
                    "$group":
                        {
                            "_id": "$" + "case_status",
                            "totalCount": {"$sum": 1}
                        }
                }
            ]
            count_json = {CaseStatus.DELETED: 0,
                          CaseStatus.OPEN: 0,
                          CaseStatus.CLOSED: 0,
                          CaseStatus.RESOLVED: 0,
                          CaseStatus.REPLIED: 0,
                          CaseStatus.ASSIGNED: 0,
                          CaseStatus.RE_ASSIGNED: 0
                          }
            case_data = self.mongo_obj.aggregate_query(json_data=json_data,
                                                       database_name=DBMapping.support_lens_configuration,
                                                       collection_name=DBMapping.ticket_configuration)
            for each_case in case_data:
                count_json.update({each_case["_id"]: each_case["totalCount"]})
            for header in count_json_header_content:
                if header["key"] in count_json:
                    header["count"] = count_json[header["key"]]
            final_json["data"] = deepcopy(count_json_header_content)
        except Exception as e:
            logger.exception(str(e))
        return final_json["data"]

    def get_dashboard_category_details(self, input_json):
        final_json = dict(status=StatusMessages.FAILED, message=StatusMessages.FAILED, data=dict())
        try:
            record_data = list(self.mongo_obj.aggregate(
                db_name=DBMapping.support_lens_configuration,
                collection_name=DBMapping.ticket_configuration,
                list_for_aggregation=[
                    {
                        '$match': {
                            'project_id': input_json["project_id"],
                            'case_status': {
                                '$in': input_json["tab_name"]
                            }
                        }
                    }]))
            return_json = list()
            category_json = dict()
            users_json = dict()
            for record in record_data:
                if record["CategoryName"] not in category_json:
                    category_json.update({record["CategoryName"]: 0})
                category_json[record["CategoryName"]] = category_json[record["CategoryName"]] + 1

            for record in record_data:
                if "assigned_to" not in record:
                    continue
                if record["assigned_to"] not in users_json:
                    users_json.update({record["assigned_to"]: 0})
                users_json[record["assigned_to"]] = users_json[record["assigned_to"]] + 1
            if input_json["progress_type"] == "engineerWise":
                users_data = list(self.mongo_obj.aggregate(
                    db_name=DBMapping.ilens_configuration,
                    collection_name=DBMapping.user,
                    list_for_aggregation=[
                        {
                            '$match': {
                                'user_id': {"$in": list(users_json.keys())}
                            }
                        }
                    ]))
                users_list = dict()
                for user in users_data:
                    users_list.update({user["user_id"]: user["username"]})

                for key, value in users_json.items():
                    return_json.append({
                        "name": users_list[key],
                        "subTitle": f"{value} Cases",
                        "bars": [
                            {
                                "value": value * 10,
                                "color": "#FFCA28",
                                "label": f"{value} Cases"
                            }
                        ]
                    })
                final_json = dict(status=StatusMessages.SUCCESS, message=StatusMessages.SUCCESS,
                                  data=dict(engineerWise=deepcopy(return_json)))
            else:
                for key, value in category_json.items():
                    return_json.append({
                        "name": key,
                        "subTitle": f"{value} Cases",
                        "bars": [
                            {
                                "value": value * 10,
                                "color": "#FFCA28",
                                "label": f"{value} Cases"
                            }
                        ]
                    })
                final_json = dict(status=StatusMessages.SUCCESS, message=StatusMessages.SUCCESS,
                                  data=dict(categoryWise=deepcopy(return_json)))


        except Exception as e:
            logger.exception(str(e))
        return final_json
