import time
from copy import deepcopy

from scripts.config.app_constants import StatusMessages, StaticJsons, DBMapping
from scripts.config.db_connection_obj import ConnectionObj
from scripts.logging.logger import logger
from scripts.utils.get_new_id import GetNewId
from scripts.utils.mongo_utility import MongoConnect


class CategoryConfigurationHandler:
    def __init__(self):
        try:
            logger.debug("Inside the Support lens Handler module")
            self.new_id = GetNewId()
            self.mongo_obj = ConnectionObj.mongo_connection_obj
            if not ConnectionObj.mongo_connection_obj:
                self.mongo_obj = ConnectionObj.mongo_connection_obj = MongoConnect()
        except Exception as e:
            logger.exception("Exception in the data utility definition" + str(e))

    def get_category_list(self, input_json):
        final_json = {"status": StatusMessages.FAILED,
                      "data": dict(headerContent=StaticJsons.CATEGORY_HEADERCONTENT, bodyContent=list()),
                      "message": StatusMessages.CATEGORY_LIST}
        logger.debug("Inside get_category_list definition")
        try:
            records = list(self.mongo_obj.aggregate(
                db_name=DBMapping.support_lens_configuration,
                collection_name=DBMapping.category_configuration,
                list_for_aggregation=[
                    {
                        '$match': {
                            'project_id': input_json["project_id"]
                        }
                    }, {
                        '$project': {
                            '_id': 0,
                            'category_id': '$category_id',
                            'category_name': '$CategoryName',
                            "sub_categories": '$SubCategories',
                            'description': '$Description'
                        }
                    }
                ]))
            for each_record in records:
                if "sub_categories" in each_record:
                    sub_categories = [item["subCategoryName"] for item in each_record["sub_categories"]]
                    each_record["sub_categories"] = ",".join(sub_categories)
            final_json["data"]["bodyContent"] = deepcopy(records)
            final_json["status"] = StatusMessages.SUCCESS
            final_json["message"] = StatusMessages.SUCCESS
        except Exception as e:
            logger.debug("Exception occurred while fetching category list data" + str(e))
        return final_json

    def fetch_category_data(self, input_json):
        final_json = {"status": StatusMessages.FAILED,
                      "data": dict()}
        logger.debug("Inside fetch_category_data definition")
        try:
            record_data = list(self.mongo_obj.aggregate(
                db_name=DBMapping.support_lens_configuration,
                collection_name=DBMapping.category_configuration,
                list_for_aggregation=[
                    {
                        '$match': {
                            'project_id': input_json["project_id"],
                            "category_id": input_json["category_id"]
                        }
                    }, {
                        '$project': {
                            '_id': 0,
                            'category_name': '$CategoryName',
                            "sub_categories": '$SubCategories',
                            'description': '$Description',
                            'category_id': '$category_id',
                            'project_id': '$project_id'
                        }
                    }
                ]))
            records = list()
            if len(record_data):
                records = record_data[0]
                for data in records["sub_categories"]:
                    data['label'] = data.pop("subCategoryName")
            final_json["data"] = deepcopy(records)
            sub_category_list = self.fetch_subcategory_data(project_id=input_json["project_id"])
            if sub_category_list:
                final_json["data"]["total_sub_categories"] = deepcopy(sub_category_list)
            final_json["status"] = StatusMessages.SUCCESS
        except Exception as e:
            logger.debug("Exception occurred while fetching category data" + str(e))
        return final_json

    def save_category_data(self, input_json):
        final_json = dict(status=StatusMessages.FAILED, message=StatusMessages.CATEGORY_DATA)
        logger.debug("Inside save_category_data definition")
        try:
            for data in input_json["sub_categories"]:
                data['subCategoryName'] = data.pop("label")
                if "sub_category_id" not in data or data["sub_category_id"] in [str(), ""]:
                    sub_category_id = self.save_subCategory_data(label_name=data["subCategoryName"],
                                                                 project_id=input_json["project_id"])
                    if not sub_category_id:
                        return final_json
                    data["sub_category_id"] = sub_category_id
            user = self.new_id.get_user_id()
            if "user_id" in user:
                user = user["user_id"]
            if input_json["type"] == "save":
                insert_json = dict(CategoryName=input_json["category_name"], Description=input_json["description"],
                                   SubCategories=input_json["sub_categories"],
                                   category_id="category_" + self.new_id.get_next_id("category"),
                                   project_id=input_json["project_id"],
                                   created_on=time.time(),
                                   created_by=user,
                                   last_updated_on=time.time(),
                                   last_updated_by=user
                                   )
                self.mongo_obj.insert_one(database_name=DBMapping.support_lens_configuration,
                                          collection_name=DBMapping.category_configuration,
                                          json_data=insert_json)
            if input_json["type"] == "edit":
                category_data = list(self.mongo_obj.aggregate(
                    db_name=DBMapping.support_lens_configuration,
                    collection_name=DBMapping.category_configuration,
                    list_for_aggregation=[
                        {
                            '$match': {
                                'category_id': input_json["category_id"]
                            }
                        }
                    ]))
                if len(category_data):
                    update_json = category_data[0]
                    update_json.update(CategoryName=input_json["category_name"], Description=input_json["description"],
                                       SubCategories=input_json["sub_categories"],
                                       category_id=input_json["category_id"],
                                       project_id=input_json["project_id"],
                                       last_updated_on=time.time(),
                                       last_updated_by=user
                                       )
                    query_json = dict(project_id=input_json["project_id"], category_id=input_json["category_id"])
                    self.mongo_obj.update_one(db_name=DBMapping.support_lens_configuration,
                                              collection_name=DBMapping.category_configuration,
                                              set_json=update_json, query=query_json)

            final_json = dict(status=StatusMessages.SUCCESS, message="Category saved Successfully")
        except Exception as e:
            logger.debug("Exception occurred while saving category data" + str(e))
        return final_json

    def save_subCategory_data(self, label_name, project_id):
        try:
            insert_json = dict(subCategoryName=label_name,
                               sub_category_id="sub_category_" + self.new_id.get_next_id("sub_category"),
                               project_id=project_id,
                               created_on=time.time(),
                               created_by=self.new_id.get_user_id())
            self.mongo_obj.insert_one(database_name=DBMapping.support_lens_configuration,
                                      collection_name=DBMapping.subcategory_configuration,
                                      json_data=insert_json)
            return insert_json["sub_category_id"]
        except Exception as e:
            logger.debug("Exception occurred while saving save_subCategory_data" + str(e))
        return False

    def delete_category_data(self, input_json):
        final_json = dict(status=StatusMessages.FAILED, message=StatusMessages.CATEGORY_DELETE)
        try:
            query_json = dict(project_id=input_json["project_id"], category_id=input_json["category_id"])
            self.mongo_obj.delete_one_record(db_name=DBMapping.support_lens_configuration,
                                             collection_name=DBMapping.category_configuration,
                                             query_json=query_json)
            # self.mongo_obj.delete_many_with_filter(db_name=DBMapping.support_lens_configuration,
            #                                        collection_name=DBMapping.resolver_configuration,
            #                                        filter_query=query_json)
            final_json = dict(status=StatusMessages.SUCCESS, message="Category deleted Successfully")
        except Exception as e:
            logger.debug("Exception occurred while deleting category data" + str(e))
        return final_json

    def fetch_subcategory_data(self, project_id):
        try:
            records = list(self.mongo_obj.aggregate(
                db_name=DBMapping.support_lens_configuration,
                collection_name=DBMapping.subcategory_configuration,
                list_for_aggregation=[
                    {
                        '$match': {
                            'project_id': project_id
                        }
                    }, {
                        '$project': {
                            '_id': 0,
                            'value': '$sub_category_id',
                            'label': '$subCategoryName'
                        }
                    }
                ]))
            return records
        except Exception as e:
            logger.debug("Exception while fetching sub category details:" + str(e))
            return False
