import base64
import os
import time
from copy import deepcopy
from datetime import datetime

from scripts.config import app_configuration
from scripts.config.app_constants import DBMapping, CaseStatus, StatusMessages, FILE_PATH, StaticJsons
from scripts.config.db_connection_obj import ConnectionObj
from scripts.logging.logger import logger
from scripts.utils.get_new_id import GetNewId
from scripts.utils.mongo_utility import MongoConnect


class TicketSetupHandler:
    def __init__(self):
        try:
            logger.debug("Inside the Resolver setup module")
            self.new_id = GetNewId()
            self.mongo_obj = ConnectionObj.mongo_connection_obj
            if not ConnectionObj.mongo_connection_obj:
                self.mongo_obj = ConnectionObj.mongo_connection_obj = MongoConnect()
        except Exception as e:
            logger.exception("Exception in the data utility definition" + str(e))

    def save_ticket_data(self, input_json):
        final_json = {"status": StatusMessages.FAILED, "message": StatusMessages.FAILED}
        try:
            logger.debug("Inside save ticket data definition")
            category_data = list(self.mongo_obj.aggregate(
                db_name=DBMapping.support_lens_configuration,
                collection_name=DBMapping.category_configuration,
                list_for_aggregation=[
                    {
                        '$match': {
                            'category_id': input_json["category"]
                        }
                    }, {
                        '$project': {
                            '_id': 0,
                            "CategoryName": "$CategoryName"
                        }
                    }
                ]))
            project_data = self.mongo_obj.find_one(db_name=DBMapping.ilens_configuration,
                                                   collection_name=DBMapping.customer_projects,
                                                   query={"customer_project_id": input_json["project_id"]})
            subcategory_data = list(self.mongo_obj.aggregate(
                db_name=DBMapping.support_lens_configuration,
                collection_name=DBMapping.subcategory_configuration,
                list_for_aggregation=[
                    {
                        '$match': {
                            'sub_category_id': input_json["subCategory"]
                        }
                    }, {
                        '$project': {
                            '_id': 0,
                            "subCategoryName": "$subCategoryName"
                        }
                    }
                ]))
            site_data = list(self.mongo_obj.aggregate(
                db_name=DBMapping.ilens_configuration,
                collection_name=DBMapping.site_conf,
                list_for_aggregation=[
                    {
                        '$match': {
                            'site_id': input_json["site"]
                        }
                    }, {
                        '$project': {
                            '_id': 0
                        }
                    }
                ]))
            # user_data = list(self.mongo_obj.aggregate(
            #     db_name=DBMapping.ilens_configuration,
            #     collection_name=DBMapping.user,
            #     list_for_aggregation=[
            #         {
            #             '$match': {
            #                 'user_id': {"$in": input_json["contacts"]}
            #             }
            #         }, {
            #             '$project': {
            #                 '_id': 0
            #             }
            #         }
            #     ]))
            if len(category_data):
                category_data = category_data[0]
            else:
                final_json = dict(message="Invalid category selected", status=StatusMessages.FAILED)
                return final_json
            if len(subcategory_data):
                subcategory_data = subcategory_data[0]
            else:
                final_json = dict(message="Invalid subcategory selected", status=StatusMessages.FAILED)
                return final_json
            if len(site_data):
                site_data = site_data[0]
            else:
                final_json = dict(message="Invalid Site selected", status=StatusMessages.FAILED)
                return final_json
            user = self.new_id.get_user_id()
            if "user_id" in user:
                user_data = self.mongo_obj.find_one(db_name=DBMapping.ilens_configuration,
                                                    collection_name=DBMapping.user,
                                                    query={"user_id": user["user_id"]})
                user = dict(userName=user_data["username"], user_id=user)
            if input_json["type"] == "create":
                case_id = "case_" + self.new_id.get_next_id("case")
            else:
                case_id = input_json["case_id"]
            if "deletedList" in input_json and len(input_json["deletedList"]):
                for file in input_json["deletedList"]:
                    response = self.delete_file_details(file, input_json["case_id"])
            if len(input_json["fileNameBlock"]):
                for file_data in input_json["fileNameBlock"]:
                    if "attachment" not in file_data:
                        continue
                    response = self.save_file_details(file_data, case_id)
                    del file_data["attachment"]
            if input_json["type"].lower() == "create":
                insert_json = dict(CategoryName=category_data["CategoryName"],
                                   ProjectName=project_data["customer_project_name"],
                                   category_id=input_json["category"],
                                   subCategoryName=subcategory_data["subCategoryName"],
                                   sub_category_id=input_json["subCategory"],
                                   Description=input_json["description"],
                                   subject=input_json["subject"],
                                   site_id=input_json["site"],
                                   siteName=site_data["site_name"],
                                   case_id=case_id,
                                   project_id=input_json["project_id"],
                                   case_status=CaseStatus.OPEN,
                                   created_on=time.time(),
                                   created_by=user,
                                   file_details=input_json["fileNameBlock"],
                                   last_updated_on=time.time(),
                                   last_updated_by=user,
                                   user_details=input_json["contacts"]
                                   )
                self.mongo_obj.insert_one(database_name=DBMapping.support_lens_configuration,
                                          collection_name=DBMapping.ticket_configuration,
                                          json_data=insert_json)
            if input_json["type"].lower() == "edit":
                query_json = {"case_id": input_json["case_id"]}
                case_data = self.mongo_obj.find_one(db_name=DBMapping.support_lens_configuration,
                                                    collection_name=DBMapping.ticket_configuration,
                                                    query=query_json,
                                                    search_json={"_id": 0})
                if len(case_data):
                    case_data.update(CategoryName=category_data["CategoryName"],
                                     category_id=input_json["category"],
                                     subCategoryName=subcategory_data["subCategoryName"],
                                     sub_category_id=input_json["subCategory"],
                                     Description=input_json["description"],
                                     subject=input_json["subject"],
                                     site_id=input_json["site"],
                                     siteName=site_data["site_name"],
                                     project_id=input_json["project_id"],
                                     case_status=CaseStatus.OPEN,
                                     file_details=input_json["fileNameBlock"],
                                     last_updated_on=time.time(),
                                     last_updated_by=user,
                                     user_details=input_json["contacts"]
                                     )
                    self.mongo_obj.update_one(db_name=DBMapping.support_lens_configuration,
                                              collection_name=DBMapping.ticket_configuration,
                                              set_json=case_data, query=query_json)
            final_json = dict(status=StatusMessages.SUCCESS, message="Casedata saved successfully")
        except Exception as e:
            logger.exception("Exception while saving ticket data" + str(e))
        return final_json

    def save_file_details(self, input_json, case_id):
        try:
            file_save_path = self.check_file_extenstions(input_json["name"], case_id)
            if not file_save_path:
                return False
            decoded_data = base64.b64decode(str(input_json["attachment"]))
            if not os.path.exists(file_save_path):
                os.makedirs(file_save_path)
            with open(f"{file_save_path}/{input_json['name']}", "wb") as file:
                file.write(decoded_data)
                file.close()
            return True
        except Exception as e:
            logger.exception("Exception while saving ticket data" + str(e))
            return False

    def delete_file_details(self, file_name, case_id):
        try:
            file_delete_path = self.check_file_extenstions(file_name, case_id)
            if not file_delete_path:
                return False
            if os.path.exists(f"{file_delete_path}\{file_name}"):
                os.remove(f"{file_delete_path}\{file_name}")
            return True
        except Exception as e:
            logger.exception("Exception while saving ticket data" + str(e))
            return False

    def check_file_extenstions(self, file_name, case_id):
        file_extenstion = file_name.split(".")[-1].lower()
        if file_extenstion in ["csv"]:
            file_save_path = os.path.join(app_configuration.FILES_SAVE_PATH, FILE_PATH.CSV_PATH, case_id)
        elif file_extenstion in ["jpg", "jpeg", "png", "svg"]:
            file_save_path = os.path.join(app_configuration.FILES_SAVE_PATH, FILE_PATH.IMAGES, case_id)
        elif file_extenstion in ["txt"]:
            file_save_path = os.path.join(app_configuration.FILES_SAVE_PATH, FILE_PATH.TXT, case_id)
        elif file_extenstion in ["pdf"]:
            file_save_path = os.path.join(app_configuration.FILES_SAVE_PATH, FILE_PATH.PDF, case_id)
        elif file_extenstion in ["docs", "docx", "doc"]:
            file_save_path = os.path.join(app_configuration.FILES_SAVE_PATH, FILE_PATH.DOCS, case_id)
        else:
            return False
        return file_save_path

    def fetch_ticket_details(self, input_json):
        final_json = dict(status=StatusMessages.FAILED, message=StatusMessages.FAILED, data=list())
        try:
            response_data = list(self.mongo_obj.aggregate(
                db_name=DBMapping.support_lens_configuration,
                collection_name=DBMapping.ticket_configuration,
                list_for_aggregation=[
                    {
                        '$match': {
                            'case_id': input_json["case_id"]
                        }
                    }, {
                        '$project': {
                            '_id': 0,
                            'category': "$category_id",
                            'subCategory': '$sub_category_id',
                            'site': '$site_id',
                            'subject': "$subject",
                            'fileNameBlock': '$file_details',
                            'description': '$Description',
                            'contacts': '$user_details',
                        }
                    }
                ]))
            if len(response_data):
                response_data = response_data[0]
            final_json = dict(status=StatusMessages.SUCCESS, message=StatusMessages.SUCCESS,
                              data=deepcopy(response_data))

        except Exception as e:
            logger.exception("Exception while fetching case details" + str(e))
        return final_json

    def delete_case_details(self, input_json):
        try:
            if "case_id" in input_json and input_json["case_id"] != "":
                query = {"case_id": input_json["case_id"]}
                new_values = {"$set": {"case_status": CaseStatus.DELETE}}
                ConnectionObj.mongo_connection_obj.update_one(db_name=DBMapping.support_lens_configuration,
                                                              collection_name=DBMapping.ticket_configuration,
                                                              query=query, set_json=new_values)
            return {"status": "success", "message": "Case deleted successfully"}
        except Exception as e:
            logger.exception(str(e))
            return {"status": "failed", "message": str(e)}

    def get_support_case_table_details(self, input_json):
        try:
            final_json = dict(status=StatusMessages.FAILED, message=StatusMessages.FAILED,
                              data=dict(tableData=dict(headerContent=StaticJsons.SUPPORTLENS_FETCHTABLE_HEADERCONTENT),
                                        tableActions=StaticJsons.SUPPORTLENS_FETCHTABLE_TABLEACTIONS,
                              enableRowExpand=True, table_type="infinite_scroll", hideSearch=True, server_search=True))

            records = list(self.mongo_obj.aggregate(
                db_name=DBMapping.support_lens_configuration,
                collection_name=DBMapping.ticket_configuration,
                list_for_aggregation=[
                    {
                        '$match': {
                            'project_id': input_json["project_id"]
                        }
                    }, {
                        '$project': {
                            '_id': 0,
                            'case_number': '$case_id',
                            'status': '$case_status',
                            "subject": '$subject',
                            "customer_project_id": "$ProjectName",
                            'last_updated': '$last_updated_on',
                            'created_by': '$created_by'
                        }
                    }
                ]))
            for record in records:
                record["created_by"] = record["created_by"]["userName"]
                record["last_updated"] = datetime.fromtimestamp(record["last_updated"]).strftime("%d %b %Y,%H:%M")
                record["expandData"] = [{"label": str(key).replace("_", " ").replace("-", " ").title(), "value": value}
                                        for key, value in record.items()]

            final_json.update(status=StatusMessages.SUCCESS, message=StatusMessages.SUCCESS)
            final_json["data"]["tableData"].update(bodyContent=deepcopy(records))
            final_json["data"]["total_no"] = len(records)
        except Exception as e:
            logger.exception(str(e))
        return final_json
