import time
from copy import deepcopy

from scripts.config.app_constants import StatusMessages, StaticJsons, DBMapping
from scripts.config.db_connection_obj import ConnectionObj
from scripts.logging.logger import logger
from scripts.utils.get_new_id import GetNewId
from scripts.utils.mongo_utility import MongoConnect


class ResolverSetupHandler:
    def __init__(self):
        try:
            logger.debug("Inside the Resolver setup module")
            self.new_id = GetNewId()
            self.mongo_obj = ConnectionObj.mongo_connection_obj
            if not ConnectionObj.mongo_connection_obj:
                self.mongo_obj = ConnectionObj.mongo_connection_obj = MongoConnect()
        except Exception as e:
            logger.exception("Exception in the data utility definition" + str(e))

    def get_resolver_list(self, input_json):
        final_json = {"status": StatusMessages.FAILED,
                      "data": dict(headerContent=StaticJsons.RESOLEVER_HEADERCONTENT, bodyContent=list())}
        logger.debug("Inside get_category_list definition")
        try:
            records = list(self.mongo_obj.aggregate(
                db_name=DBMapping.support_lens_configuration,
                collection_name=DBMapping.resolver_configuration,
                list_for_aggregation=[
                    {
                        '$match': {
                            'project_id': input_json["project_id"]
                        }
                    }, {
                        '$project': {
                            '_id': 0,
                            'resolver_group_id': '$resolver_group_id',
                            'resolver_group_name': '$ResolverGroupname',
                            "sub_categories": '$SubCategories',
                            "category": "$CategoryName",
                            'description': '$Description'
                        }
                    }
                ]))
            for each_record in records:
                if "sub_categories" in each_record:
                    sub_categories = [item["sub_CategoryName"] for item in each_record["sub_categories"]]
                    each_record["sub_categories"] = ",".join(sub_categories)
            final_json["data"]["bodyContent"] = deepcopy(records)
            final_json["status"] = StatusMessages.SUCCESS
        except Exception as e:
            logger.debug("Exception occurred while fetching resolver list data" + str(e))
        return final_json

    def save_resolver_data(self, input_json):
        final_json = dict(status=StatusMessages.FAILED, message=StatusMessages.RESOLVER_DATA)
        logger.debug("Inside save_resolver_data definition")
        try:
            for data in input_json["sub_categories"]:
                data['sub_CategoryName'] = data.pop("label")
            category_data = list(self.mongo_obj.aggregate(
                db_name=DBMapping.support_lens_configuration,
                collection_name=DBMapping.category_configuration,
                list_for_aggregation=[
                    {
                        '$match': {
                            'category_id': input_json["category"]
                        }
                    }, {
                        '$project': {
                            '_id': 0,
                            "category": "$CategoryName"
                        }
                    }
                ]))
            if len(category_data):
                category_data = category_data[0]
            user = self.new_id.get_user_id()
            if "user_id" in user:
                user = user["user_id"]
            if input_json["type"] == "save":
                insert_json = dict(ResolverGroupname=input_json["resolver_group_name"],
                                   Description=input_json["description"],
                                   SubCategories=input_json["sub_categories"],
                                   resolver_group_id="resolver_group_" + self.new_id.get_next_id("resolver_group"),
                                   category_id=input_json["category"],
                                   CategoryName=category_data.get("category", None),
                                   project_id=input_json["project_id"],
                                   created_on=time.time(),
                                   created_by=user,
                                   last_updated_on=time.time(),
                                   last_updated_by=user
                                   )
                self.mongo_obj.insert_one(database_name=DBMapping.support_lens_configuration,
                                          collection_name=DBMapping.resolver_configuration,
                                          json_data=insert_json)
            if input_json["type"] == "edit":
                resolver_data = list(self.mongo_obj.aggregate(
                    db_name=DBMapping.support_lens_configuration,
                    collection_name=DBMapping.resolver_configuration,
                    list_for_aggregation=[
                        {
                            '$match': {
                                'resolver_group_id': input_json["resolver_group_id"]
                            }
                        }]))
                for record in resolver_data:
                    record.update(CategoryName=category_data.get("category", None), Description=input_json["description"],
                                  SubCategories=input_json["sub_categories"],
                                  category_id=input_json["category"],
                                  project_id=input_json["project_id"],
                                  last_updated_on=time.time(),
                                  last_updated_by=user
                                  )
                    query_json = dict(project_id=input_json["project_id"], resolver_group_id=input_json["resolver_group_id"])
                    self.mongo_obj.update_one(db_name=DBMapping.support_lens_configuration,
                                              collection_name=DBMapping.resolver_configuration,
                                              set_json=record, query=query_json)

            final_json = dict(status=StatusMessages.SUCCESS, message="Category saved Successfully")
        except Exception as e:
            logger.debug("Exception occurred while saving category data" + str(e))
        return final_json

    def fetch_resolver_data(self, input_json):
        final_json = {"status": StatusMessages.FAILED,
                      "data": dict()}
        logger.debug("Inside fetch_category_data definition")
        try:
            records = list(self.mongo_obj.aggregate(
                db_name=DBMapping.support_lens_configuration,
                collection_name=DBMapping.resolver_configuration,
                list_for_aggregation=[
                    {
                        '$match': {
                            'project_id': input_json["project_id"],
                            "resolver_group_id": input_json["resolver_group_id"]
                        }
                    }, {
                        '$project': {
                            '_id': 0,
                            "sub_categories": '$SubCategories',
                            'description': '$Description',
                            'resolver_group_name': '$ResolverGroupname',
                            'category': '$category_id',
                            "resolver_group_id": "$resolver_group_id"
                        }
                    }
                ]))
            if len(records):
                records = records[0]
                for data in records["sub_categories"]:
                    data['label'] = data.pop("sub_CategoryName")
            final_json["data"] = deepcopy(records)
            final_json["status"] = StatusMessages.SUCCESS
        except Exception as e:
            logger.debug("Exception occurred while fetching reslover data" + str(e))
        return final_json

    def delete_resolver_data(self, input_json):
        final_json = dict(status=StatusMessages.FAILED, message=StatusMessages.RESOLVER_DELETE)
        try:
            query_json = dict(project_id=input_json["project_id"], resolver_group_id=input_json["resolver_group_id"])
            self.mongo_obj.delete_one_record(db_name=DBMapping.support_lens_configuration,
                                             collection_name=DBMapping.resolver_configuration,
                                             query_json=query_json)
            final_json = dict(status=StatusMessages.SUCCESS, message="Resolved deleted Successfully")
        except Exception as e:
            logger.debug("Exception occurred while deleting Resolver data" + str(e))
        return final_json

    def get_category_metadata(self, input_json):
        final_json = {"status": StatusMessages.FAILED,
                      "data": dict(),
                      "message": StatusMessages.CATEGORY_FETCH}
        logger.debug("Inside get category metadata definition")
        try:
            category_data = list(self.mongo_obj.aggregate(
                db_name=DBMapping.support_lens_configuration,
                collection_name=DBMapping.category_configuration,
                list_for_aggregation=[
                    {
                        '$match': {
                            'project_id': input_json["project_id"]
                        }
                    }, {
                        '$project': {
                            '_id': 0,
                            "label": "$CategoryName",
                            "value": "$category_id",
                            "sub_categories": "$SubCategories"
                        }
                    }
                ]))
            category_list = list()
            sub_category_list = dict()
            for category in category_data:
                category_list.append(dict(label=category["label"], value=category["value"]))
                if category["value"] not in sub_category_list:
                    sub_category_list.update({category["value"]: list()})
                    for data in category["sub_categories"]:
                        data['label'] = data.pop("subCategoryName")
                sub_category_list[category["value"]] = category["sub_categories"]

            final_json.update(status=StatusMessages.SUCCESS, message=StatusMessages.SUCCESS,
                              data=dict(categories=deepcopy(category_list), sub_categories=deepcopy(sub_category_list)))

        except Exception as e:
            logger.debug("Exception occurred while deleting Resolver data" + str(e))
        return final_json
