"""
This file exposes configurations from config file and environments as Class Objects
"""
import shutil

if __name__ == '__main__':
    from dotenv import load_dotenv

    load_dotenv()
import os.path
import sys
from configparser import ConfigParser, BasicInterpolation


class EnvInterpolation(BasicInterpolation):
    """
    Interpolation which expands environment variables in values.
    """

    def before_get(self, parser, section, option, value, defaults):
        value = super().before_get(parser, section, option, value, defaults)

        if not os.path.expandvars(value).startswith('$'):
            return os.path.expandvars(value)
        else:
            return


try:
    config = ConfigParser(interpolation=EnvInterpolation())
    config.read(f"conf/application.conf")
except Exception as e:
    print(f"Error while loading the config: {e}")
    print("Failed to Load Configuration. Exiting!!!")
    sys.stdout.flush()
    sys.exit()


class Service:
    HOST = config.get("SERVICE", "host")
    PORT = config.getint("SERVICE", "port")

class KafkaConf:
    host = config.get('KAFKA', 'host')
    port = config.get('KAFKA', 'port')
    topic = config.get('KAFKA', 'topic')
    if not any([topic, host, port]):
        print("KAFKA env variables missing, continuing without Kafka/Kairos support")
    audit_topic = config.get('KAFKA', 'audit_topic')
    enable_sites_partition = config.getboolean("KAFKA", "ENABLE_KAFKA_PARTITION", fallback=True)
    split_key = config["KAFKA"].get('KAFKA_PARTITION_KEY', 'site_id')
    round_robin_enable = config.getboolean("KAFKA", "ROUND_ROBIN_PARTITION", fallback=True)


class StoragePaths:
    module_name = config.get('SERVICE', 'module_name')
    if not module_name:
        module_name = "form_management"
    base_path = os.path.join("data", module_name)


class PathToStorage:
    BASE_PATH = config.get("DIRECTORY", "base_path")
    if not BASE_PATH:
        print("Error, environment variable BASE_PATH not set")
        sys.exit(1)
    MOUNT_DIR = config.get("DIRECTORY", "mount_dir")
    if not MOUNT_DIR:
        print("Error, environment variable MOUNT_DIR not set")
        sys.exit(1)
    MODULE_PATH = os.path.join(BASE_PATH, MOUNT_DIR.lstrip('/'))
    FORM_IO_UPLOADS = os.path.join(MODULE_PATH, "form_io_uploads")
    TEMPLATES_UPLOADS = os.path.join(MODULE_PATH, "templates_uploads")
    LOGS_MODULE_PATH = f"{BASE_PATH}/logs{MOUNT_DIR}/"


class RedisConfig:
    host = config["REDIS"]["host"]
    port = int(config["REDIS"]["port"])
    login_db = config["REDIS"]["login_db"]
    project_tags_db = config.getint("REDIS", "project_tags_db")

