import string
from contextlib import suppress
from copy import deepcopy, copy
from typing import Iterable, List, Dict

import jinja2
import openpyxl
import openpyxl.cell
from openpyxl import load_workbook, Workbook
from openpyxl.formula.translate import Translator
from openpyxl.utils.cell import coordinate_from_string, column_index_from_string, get_column_letter, _get_column_letter
from openpyxl.worksheet.cell_range import CellRange

from scripts.logging.logging import logger


class ExcelReportRender:

    def write_data_to_template(
            self,
            data: Dict,
            template: str, output: str,
            sheet_name: str = 'Sheet1',
            fill_range: str = "A1:N100"
    ):
        __wb__ = load_workbook(filename=template)
        __sheet__ = __wb__[sheet_name]
        __start_coordinates__, __end_coordinates__ = self.get_row_column(fill_range)
        start_row, start_column = __start_coordinates__
        end_row, end_column = __end_coordinates__

        for i in range(start_row, end_row + 1):
            for j in range(start_column, end_column + 1):
                try:
                    cell_value = __sheet__.cell(row=i, column=j).value
                    if not cell_value or not isinstance(cell_value, str):
                        continue

                    if cell_value.endswith(" }}") and cell_value.startswith("{{ "):
                        __sheet__.cell(row=i, column=j).value = self.render_into_cell(cell_value, data)

                except Exception as e:
                    logger.exception(e)
        __wb__.save(output)
        return output

    @staticmethod
    def render_into_cell(cell_value, data):
        template = jinja2.Template(cell_value)
        try:
            cell_value = template.render(data)
        except Exception as e:
            logger.debug(e.args, exc_info=False)
            return ""

        with suppress(ValueError):
            cell_value = float(cell_value)

        if cell_value and isinstance(cell_value, str) and \
                cell_value.endswith(" }}") and cell_value.startswith("{{ "):
            cell_value = ""
        return cell_value

    @staticmethod
    def work_book_loader(workbooks: Iterable, sheet_name: str = 'Sheet1'):
        sheets_list = []
        for wb in workbooks:
            __wb__ = load_workbook(wb)
            sheets_list.append(__wb__[sheet_name])
        return sheets_list

    def copy_sheet_cells(
            self,
            sheet,
            start_row_child, end_row_child,
            start_column_child, end_column_child,
            formula_dict, range_selected, style_range
    ):
        # Loops through selected Rows
        for i in range(start_row_child, end_row_child + 1, 1):
            # Appends the row to a RowSelected list
            row_selected = []
            row_style = []
            for j in range(start_column_child, end_column_child + 1, 1):
                row_selected.append(sheet.cell(row=i, column=j).value)
                if sheet.cell(row=i, column=j).has_style:
                    row_style.append(sheet.cell(row=i, column=j)._style)
                x = sheet.cell(row=i, column=j).value
                if x and isinstance(x, str) and x.startswith("="):
                    coord = self.num_to_coord(i, j)
                    formula_dict[coord] = x
            # Adds the RowSelected List and nests inside the rangeSelected
            range_selected.append(row_selected)
            style_range.append(row_style)

    def paste_sheet_cells(
            self,
            master_sheet,
            start_row_paste, end_row_paste,
            start_col_paste, end_col_paste,
            style_range, range_selected,
            diff_in_columns
    ):
        count_row = 0

        for i in range(start_row_paste, end_row_paste + 1, 1):
            count_col = 0
            for j in range(start_col_paste, end_col_paste + 1, 1):
                master_sheet.cell(row=i, column=j).value = range_selected[count_row][count_col]
                master_sheet.cell(row=i, column=j)._style = style_range[count_row][count_col]
                x = master_sheet.cell(row=i, column=j).value
                if x and isinstance(x, str) and x.startswith("="):
                    coord = self.num_to_coord(
                        count_row - 1 + diff_in_columns,
                        count_col + 1 + diff_in_columns,
                        False
                    )
                    master_sheet.cell(row=i, column=j).value = Translator(x, coord).translate_formula(
                        self.num_to_coord(i, j, False))
                count_col += 1
            count_row += 1

    def repeat_in_same_sheet(self, workbooks: List, fill_range: str, output: str, sheet_name: str = "Sheet1"):
        master_workbook = load_workbook(filename=workbooks[0])
        master_sheet = master_workbook[sheet_name]
        del workbooks[0]

        __start_coords__, __end_coords__ = self.get_row_column(fill_range)
        start_row, start_column = __start_coords__
        end_row, end_column = __end_coords__

        workbook_data = self.work_book_loader(workbooks, sheet_name=sheet_name)
        for sheet in workbook_data:
            start_coords_child, end_coord_child = self.get_row_column(fill_range)
            start_row_child, start_column_child = start_coords_child
            end_row_child, end_column_child = end_coord_child
            range_selected = []
            style_range = []
            formula_dict = {}
            # Loops through selected Rows
            self.copy_sheet_cells(
                sheet,
                start_row_child, end_row_child,
                start_column_child, end_column_child,
                formula_dict, range_selected, style_range
            )

            start_row_paste = start_row
            end_row_paste = end_row

            diff_in_columns = abs(start_column - end_column)
            start_col_paste = end_column + 1
            end_col_paste = start_col_paste + diff_in_columns

            self.paste_sheet_cells(
                master_sheet,
                start_row_paste, end_row_paste,
                start_col_paste, end_col_paste,
                style_range, range_selected,
                diff_in_columns
            )

            start_row = start_row_paste
            end_row = end_row_paste
            start_column = start_col_paste
            end_column = end_col_paste

        master_workbook.save(output)

    @staticmethod
    def copy_dimensions(sheet, new_sheet):
        try:
            for idx, rd in sheet.row_dimensions.items():
                new_sheet.row_dimensions[idx] = copy(rd)
            for k, cd in sheet.column_dimensions.items():
                new_sheet.column_dimensions[k].width = cd.width
            for mcr in sheet.merged_cells:
                cr = CellRange(mcr.coord)
                new_sheet.merge_cells(cr.coord)
        except Exception as e:
            logger.exception(e)

    def repeat_in_different_sheet(self, workbooks, fill_range, output, sheet_names, sheet_name: str = "Sheet1"):
        master_workbook = load_workbook(filename=workbooks[0])
        master_sheet = master_workbook[sheet_name]
        master_sheet.title = deepcopy(sheet_names[0])
        del workbooks[0]
        del sheet_names[0]

        workbook_data = self.work_book_loader(workbooks, sheet_name=sheet_name)

        for sheet, sheet_name in zip(workbook_data, sheet_names):
            master_workbook.create_sheet(sheet_name)
            new_sheet = master_workbook[sheet_name]
            for row in sheet:
                for cell in row:
                    new_sheet[cell.coordinate].value = copy(cell.value)
                    if cell.has_style:
                        new_sheet[cell.coordinate]._style = copy(cell._style)

            self.copy_dimensions(sheet, new_sheet)

        master_workbook.save(output)

    def get_row_column(self, fill_range):
        start_cell, end_cell = fill_range.split(":")
        start_row, start_column = self.coord_to_num(start_cell)
        end_row, end_column = self.coord_to_num(end_cell)
        return (start_row, start_column,), (end_row, end_column,)

    @staticmethod
    def coord_to_num(coord):
        xy = coordinate_from_string(coord)  # returns ('A',4)
        column = column_index_from_string(xy[0])
        row = xy[1]
        return row, column

    @staticmethod
    def column_num_to_string(num):
        x = _get_column_letter(num)
        return x

    @staticmethod
    def num_to_coord(row, column, zero_indexed=True):
        if zero_indexed:
            row += 1
            column += 1
        return get_column_letter(column) + str(row)

    def merge_excel_files(self, workbooks, output, sheets_dict: dict):
        master_workbook = Workbook()
        counter = 0
        sheets_list = []
        for workbook in workbooks:
            counter += 1
            __wb__ = load_workbook(workbook)
            if not __wb__.sheetnames:
                continue
            sheet_name = f"Sheet{counter}" if not sheets_dict.get(workbook) or sheets_dict.get(
                workbook) in sheets_list else sheets_dict.get(workbook)
            sheets_list.append(sheet_name)
            master_workbook.create_sheet(sheet_name)
            new_sheet = master_workbook[sheet_name]
            sheet = __wb__[__wb__.sheetnames[0]]
            self.copy_sheet(source_sheet=sheet, target_sheet=new_sheet)
        if "Sheet" in master_workbook.sheetnames and "Sheet" not in sheets_list:
            master_workbook.remove_sheet(master_workbook["Sheet"])
        master_workbook.save(output)

    def copy_sheet(self, source_sheet, target_sheet):
        self.copy_cells(source_sheet, target_sheet)  # copy all the cel values and styles
        self.copy_sheet_attributes(source_sheet, target_sheet)
        # self.copy_images(source_sheet, target_sheet)

    @staticmethod
    def copy_sheet_attributes(source_sheet, target_sheet):
        target_sheet.sheet_format = copy(source_sheet.sheet_format)
        target_sheet.sheet_properties = copy(source_sheet.sheet_properties)
        target_sheet.merged_cells = copy(source_sheet.merged_cells)
        target_sheet.page_margins = copy(source_sheet.page_margins)
        target_sheet.freeze_panes = copy(source_sheet.freeze_panes)

        # set row dimensions
        for rn in range(len(source_sheet.row_dimensions)):
            target_sheet.row_dimensions[rn] = copy(source_sheet.row_dimensions[rn])

        if source_sheet.sheet_format.defaultColWidth is None:
            print('Unable to copy default column wide')
        else:
            target_sheet.sheet_format.defaultColWidth = copy(source_sheet.sheet_format.defaultColWidth)

        # set specific column width and hidden property
        # we cannot copy the entire column_dimensions attribute so we copy selected attributes
        for key, value in source_sheet.column_dimensions.items():
            target_sheet.column_dimensions[key].min = copy(source_sheet.column_dimensions[
                                                               key].min)
            target_sheet.column_dimensions[key].max = copy(source_sheet.column_dimensions[
                                                               key].max)
            target_sheet.column_dimensions[key].width = copy(
                source_sheet.column_dimensions[key].width)  # set width for every column
            target_sheet.column_dimensions[key].hidden = copy(source_sheet.column_dimensions[key].hidden)

    @staticmethod
    def copy_cells(source_sheet, target_sheet):
        for r, row in enumerate(source_sheet.iter_rows()):
            for c, cell in enumerate(row):
                source_cell = cell
                if isinstance(source_cell, openpyxl.cell.read_only.EmptyCell):
                    continue
                target_cell = target_sheet.cell(column=c + 1, row=r + 1)
                target_cell._value = source_cell._value
                target_cell.data_type = source_cell.data_type

                if source_cell.has_style:
                    target_cell.font = copy(source_cell.font)
                    target_cell.border = copy(source_cell.border)
                    target_cell.fill = copy(source_cell.fill)
                    target_cell.number_format = copy(source_cell.number_format)
                    target_cell.protection = copy(source_cell.protection)
                    target_cell.alignment = copy(source_cell.alignment)

    def copy_images(self, source_sheet, target_sheet):
        image_loader = self.retrieve_images(sheet=source_sheet)
        try:
            for k, v in image_loader.items():
                target_sheet.add_image(v, k)
        except Exception as e:
            logger.exception(f"Unable to copy images from source sheet to target sheet {e}")

    @staticmethod
    def retrieve_images(sheet):
        _images = {}
        sheet_images = sheet._images
        for image in sheet_images:
            row = image.anchor._from.row + 1
            col = string.ascii_uppercase[image.anchor._from.col]
            _images[f'{col}{row}'] = image
        return _images
