import os.path
from datetime import datetime

from dateutil import parser
from openpyxl import load_workbook
from pytz import timezone

from scripts.logging.logging import logger
from scripts.utils.excel_render import ExcelReportRender
from scripts.utils.ilens_publish_data import DataPush

enable_timestamp = os.environ.get("CURRENT_TIMESTAMP", default="true")
code_timezone = os.environ.get("TIMEZONE", default="Asia/Kolkata")
project_id = os.environ.get("PROJECT_ID", default="project_099")


class ExcelUploadHandler:
    def __init__(self):
        self.excel = ExcelReportRender()
        self.tz = code_timezone
        self.enable_current_timestamp = True if enable_timestamp.lower() in [True, 'true'] else False
        self.project_id = project_id
        self.kafka_conn = DataPush()

    def parse_excel_data(self, template_file_path, data_file_path, template_fill_range, file_name, column_enabled,
                         sheet_name="Sheet1"):
        try:
            if not os.path.join(template_file_path):
                logger.exception(f"file_path for the file {template_file_path} not exists")
                raise FileNotFoundError
            if not os.path.join(data_file_path):
                logger.exception(f"file_path for the file {data_file_path} not exists")
                raise FileNotFoundError
            template_parameters = self.get_file_object_data(file=template_file_path, fill_range=template_fill_range,
                                                            sheet_name=sheet_name)
            data_parameters = self.get_file_object_data(file=data_file_path, fill_range=template_fill_range,
                                                        sheet_name=sheet_name)
            self.execute_data_insertion(template_parameters=template_parameters, data_parameters=data_parameters,
                                        fill_range=template_fill_range, column_enabled=column_enabled)

        except Exception as e:
            logger.exception(f"Exception occurred while parsing the excel data {e.args}")

    def get_file_object_data(self, file, fill_range, sheet_name):
        try:
            file_obj = load_workbook(filename=file)
            return_dict = {}
            sheet_names = file_obj.sheetnames
            if sheet_name not in sheet_names:
                raise
            sheet_obj = file_obj[sheet_name]
            __start_coords__, __end_coords__ = self.excel.get_row_column(fill_range)
            start_row, start_column = __start_coords__
            end_row, end_column = __end_coords__

            for i in range(start_row, end_row + 1):
                for j in range(start_column, end_column + 1):
                    cell_value = sheet_obj.cell(row=i, column=j).value
                    return_dict.update({f"{self.excel.num_to_coord(row=i, column=j, zero_indexed=False)}": cell_value})
            return return_dict

        except Exception as e:
            logger.error(e.args)
            raise

    def execute_data_insertion(self, template_parameters: dict, data_parameters: dict, fill_range,
                               column_enabled: bool):
        try:
            __start_coords__, __end_coords__ = self.excel.get_row_column(fill_range)
            start_row, start_column = __start_coords__
            end_row, end_column = __end_coords__
            columns_range = [self.excel.column_num_to_string(i) for i in range(start_column, end_column + 1)]
            rows_range = [str(i) for i in range(start_row, end_row + 1)]
            insert_data = dict()
            insert_json = {'data': {},
                           'site_id': 'site_100', 'gw_id': 'gw_1', 'pd_id': '1',
                           'retain_flag': True, 'msg_id': 1, "p_id": ""}
            key_mapping_json = dict(zip(template_parameters.values(), template_parameters.keys()))
            date_keys = [item for item in list(key_mapping_json.keys()) if item and item.startswith("date")]
            if not column_enabled:
                for _row in rows_range:
                    res = [item for item in list(template_parameters.keys()) if item.endswith(_row)]
                    _dict = {template_parameters[_element]: data_parameters[_element] for _element in res if
                             template_parameters.get(_element) and data_parameters.get(
                                 _element) and (
                                     "$" in template_parameters.get(_element) or "date" in template_parameters.get(
                                 _element))}
                    insert_data = self.prepare_data_processor_json(input_json=_dict, data_parameters=data_parameters,
                                                                   key_mapping_json=key_mapping_json,
                                                                   date_keys=date_keys, insert_data=insert_data)
            else:
                for _column in columns_range:
                    res = [item for item in list(template_parameters.keys()) if item.startswith(_column)]
                    _dict = {template_parameters[_element]: data_parameters[_element] for _element in res if
                             template_parameters.get(_element) and data_parameters.get(_element) and
                             ("$" in template_parameters.get(_element) or "date" in template_parameters.get(_element))}
                    insert_data = self.prepare_data_processor_json(input_json=_dict, data_parameters=data_parameters,
                                                                   key_mapping_json=key_mapping_json,
                                                                   date_keys=date_keys, insert_data=insert_data)
            msg_counter = 1
            for k, v in insert_data.items():
                timestamp = int(k)
                insert_json.update({
                    "data": v, "p_id": project_id, "msg_id": msg_counter, "timestamp": timestamp
                })
                logger.debug(f"Timestamp: {timestamp}, Values: {insert_data}")
                self.kafka_conn.publish_message(msg=insert_json)
                msg_counter += 1
        except Exception as e:
            logger.exception(f"Exception occurred while parsing the excel data {e.args}")

    def convert_date_string_to_timestamp(self, input_data):
        timestamp = ""
        if not input_data:
            return timestamp
        try:
            if type(input_data) is datetime:
                timestamp = int(input_data.astimezone(timezone(self.tz)).timestamp()) * 1000
            else:
                timestamp = int(parser.parse(input_data).astimezone(timezone(self.tz)).timestamp()) * 1000
        except Exception as e:
            logger.exception(f"Exception Occurred while converting the date object {e.args}")
        return timestamp

    def prepare_data_processor_json(self, input_json, data_parameters: dict, key_mapping_json: dict, date_keys: list,
                                    insert_data: dict):
        try:
            date_key = [item for item in list(input_json.keys()) if item and item.startswith("date")]
            date_key = date_key[0] if date_key else ""
            timestamp_value = self.convert_date_string_to_timestamp(input_data=input_json[date_key]) if date_key else ''
            if not timestamp_value and date_keys:
                timestamp_value = data_parameters[key_mapping_json[data_parameters[0]]]
                timestamp_value = self.convert_date_string_to_timestamp(input_data=timestamp_value)
            if not timestamp_value and self.enable_current_timestamp:
                timestamp_value = int(datetime.now().replace(hour=0, minute=0, second=0, microsecond=0).astimezone(
                    timezone(self.tz)).timestamp()) * 1000
            timestamp_str = str(timestamp_value)
            if timestamp_value not in insert_data:
                insert_data[timestamp_str] = {}
            if date_key:
                input_json.pop(date_key, None)
            insert_data[timestamp_str].update(input_json)
        except Exception as e:
            logger.exception(f"Exception occurred while json creation {e.args}")
        return insert_data
