import json
from contextlib import suppress

from ilens_kafka_publisher import KafkaPublisher
from kafka import KafkaProducer

from scripts.config.app_configurations import KafkaConf, RedisConfig
from scripts.logging.logging import logger


class DataPush:
    def __init__(self):
        try:
            self.obj = KafkaPublisher(kafka_host=KafkaConf.host,
                                      kafka_port=int(KafkaConf.port),
                                      kafka_topic=KafkaConf.topic,
                                      redis_host=RedisConfig.host,
                                      redis_port=RedisConfig.port,
                                      redis_db=RedisConfig.login_db,
                                      enable_sites_partition=KafkaConf.enable_sites_partition,
                                      split_key=KafkaConf.split_key,
                                      round_robin_enable=KafkaConf.round_robin_enable)
        except Exception as e:
            logger.error(f"Could not connect to Kafka: {e}")

    def publish_message(self, msg):
        try:
            self.obj.perform_task(msg)
        except Exception as e:
            logger.debug(f"Failed to publish message - {e}")
            logger.debug(f"Trying reconnect")


class KafkaProducerUtil:
    def __init__(self):
        try:
            self.host = KafkaConf.host
            self.port = int(KafkaConf.port)
            kafka_broker = f"{self.host}:{str(self.port)}"
            self.producer = KafkaProducer(bootstrap_servers=kafka_broker,
                                          value_serializer=lambda v: v.encode('utf-8'),
                                          api_version=(0, 10, 1))
            self.producer.flush()
        except Exception as e:
            logger.error(f"Kafka connection error: {e}")

    def publish(self, topic, data):
        try:
            kafka_response = self.producer.send(topic, data)
            logger.debug(f" Message sent to kafka with response: {kafka_response}")
            return True
        except Exception as e:
            logger.error(e)
            return False


class KairosWriter:

    def write_data(self, data_json, topic, project_id):
        kafka_conn = DataPush()
        logger.debug(f"Data being pushed to kafka topic: {topic}")
        msg_counter = 0
        for k, v in data_json.items():
            timestamp, data, site_ids = self.data_validator(k, v)
            if not data:
                continue
            for each in site_ids:
                values = {tag_id: value for tag_id, value in data.items() if tag_id.startswith(each)}
                if not values:
                    continue
                write_json = {
                    "data": values,
                    "site_id": each[:-1],
                    "gw_id": "",
                    "pd_id": "",
                    "p_id": project_id,
                    "timestamp": timestamp,
                    "msg_id": msg_counter,
                    "retain_flag": False
                }
                logger.debug(f"Timestamp: {timestamp}, Values: {data}")
                kafka_conn.publish_message(msg=write_json)
                msg_counter += 1

        return msg_counter

    @staticmethod
    def audit_data(data_json, topic):
        old_kafka_conn = KafkaProducerUtil()
        logger.debug(f"Audit Data being pushed to kafka topic: {topic}")
        msg_counter = len(data_json)
        for each in data_json:
            audit_json = dict(model="form_data_audits", record=each)
            old_kafka_conn.publish(topic, json.dumps(audit_json))
        return msg_counter

    @staticmethod
    def data_validator(timestamp, data):
        __temp__ = {}
        site_ids = set()
        for k, v in data.items():
            if not k.startswith("site"):
                continue
            if isinstance(v, int) or isinstance(v, float):
                __temp__[k] = v
                site_id = f"{k.split('$')[0]}$"
                site_ids.add(site_id)
                continue
            with suppress(ValueError):
                __temp__[k] = float(v)
                site_id = f"{k.split('$')[0]}$"
                site_ids.add(site_id)
        return int(timestamp), __temp__, site_ids
