from secrets import compare_digest

from fastapi import Response, Request, HTTPException
from fastapi.openapi.models import APIKey, APIKeyIn
from fastapi.security import APIKeyCookie
from fastapi.security.api_key import APIKeyBase

from scripts.config import Service
from scripts.constants import Secrets
from scripts.db.redis_connections import login_db
from scripts.utils.security_utils.apply_encrytion_util import create_token
from scripts.utils.security_utils.jwt_util import JWT


class CookieAuthentication(APIKeyBase):
    """
    Authentication backend using a cookie.
    Internally, uses a JWT token to store the data.
    """

    scheme: APIKeyCookie
    cookie_name: str
    cookie_secure: bool

    def __init__(
            self,
            cookie_name: str = "login-token",

    ):
        super().__init__()
        self.model: APIKey = APIKey(**{"in": APIKeyIn.cookie}, name=cookie_name)
        self.scheme_name = self.__class__.__name__
        self.cookie_name = cookie_name
        self.scheme = APIKeyCookie(name=self.cookie_name, auto_error=False)
        self.login_redis = login_db
        self.jwt = JWT()

    async def __call__(self, request: Request, response: Response) -> str:
        cookies = request.cookies
        login_token = cookies.get("login-token")
        if not login_token:
            login_token = request.headers.get("login-token")
        if not login_token:
            raise HTTPException(status_code=401)

        jwt_token = self.login_redis.get(login_token)
        if not jwt_token:
            raise HTTPException(status_code=401)

        try:
            decoded_token = self.jwt.validate(token=jwt_token)
            if not decoded_token:
                raise HTTPException(status_code=401)
        except Exception as e:
            raise HTTPException(status_code=401, detail=e.args)

        user_id = decoded_token.get("user_id")

        _token = decoded_token.get("token")

        if not compare_digest(Secrets.token, _token):
            raise HTTPException(status_code=401)
        if login_token != decoded_token.get("uid"):
            raise HTTPException(status_code=401)

        try:
            new_token = create_token(
                user_id=user_id,
                ip=request.client.host,
                token=Secrets.token,
                login_token=login_token,
            )
        except Exception as e:
            raise HTTPException(status_code=401, detail=e.args)
        response.set_cookie(
            'login-token',
            new_token,
            samesite='strict',
            secure=Service.secure_cookie,
            httponly=True,
            max_age=Secrets.LOCK_OUT_TIME_MINS * 60,
        )
        response.headers['login-token'] = new_token

        return user_id
