import traceback

from fastapi import APIRouter, Depends
from sqlalchemy.orm import Session

from scripts.constants import Endpoints, ResponseCodes
from scripts.core.handlers.api_handler import APIHandler
from scripts.core.handlers.layout_handler import LayoutHandler
from scripts.db.databases import get_db
from scripts.logging.logging import logger
from scripts.schemas.batch_oee import GetProducts, ChartRequest
from scripts.schemas.layout import GetLayoutRequest, SaveLayoutRequest
from scripts.schemas.response_models import DefaultFailureResponse, DefaultResponse
from scripts.errors import ILensError

api_handler = APIHandler()
layout_handler = LayoutHandler()
ui_service_router = APIRouter(prefix=Endpoints.api_batches, tags=["UI Services"])


@ui_service_router.post(Endpoints.api_get)
async def get_all_products(request_data: GetProducts, db: Session = Depends(get_db)):
    try:
        data = await api_handler.get_products(request_data=request_data, db=db)
        return DefaultResponse(
            data=data,
            status=ResponseCodes.SUCCESS,
            message="Products fetched successfully",
        )
    except Exception as e:
        tb = traceback.format_exc()
        logger.exception(e)
        logger.exception(tb)
        return DefaultFailureResponse(error=e.args)


@ui_service_router.post(Endpoints.api_chart_data)
async def get_chart_data(request_data: ChartRequest, db: Session = Depends(get_db)):
    try:
        data = await api_handler.get_chart_data(request_data=request_data, db=db)
        return DefaultResponse(
            data=data,
            status=ResponseCodes.SUCCESS,
            message="Chart data fetched successfully",
        )
    except ILensError as e:
        return DefaultFailureResponse(error=e.args)
    except Exception as e:
        tb = traceback.format_exc()
        logger.exception(e)
        logger.exception(tb)
        return DefaultFailureResponse(error=e.args)


@ui_service_router.post(Endpoints.get_layout)
async def get_layout(request_data: GetLayoutRequest):
    try:
        data = await layout_handler.fetch_layout(layout_request=request_data)
        return DefaultResponse(
            data=data,
            status=ResponseCodes.SUCCESS,
            message="Layout fetched successfully",
        )
    except Exception as e:
        tb = traceback.format_exc()
        logger.exception(e)
        logger.exception(tb)
        return DefaultFailureResponse(error=e.args)


@ui_service_router.post(Endpoints.save_layout)
async def save_layout(request_data: SaveLayoutRequest):
    try:
        data = await layout_handler.save_layout(layout_request=request_data)
        return DefaultResponse(
            data=data, status=ResponseCodes.SUCCESS, message="Layout saved successfully"
        )
    except Exception as e:
        tb = traceback.format_exc()
        logger.exception(e)
        logger.exception(tb)
        return DefaultFailureResponse(error=e.args)


# -------------Code Demo Backup----------------#

#
# @ui_service_router.post(Endpoints.get_batch_oee_all)
# async def get_all_batch_oee(get_oee_request: GetOEERequest,
#                             db: Session = Depends(get_db)):
#     try:
#         return await api_handler.get_oee_all(get_oee_request=get_oee_request, db=db)
#     except Exception as e:
#         tb = traceback.format_exc()
#         logger.exception(e)
#         logger.exception(tb)
#         return DefaultFailureResponse(error=e.args)
#
#
# @ui_service_router.post(Endpoints.get_batch_oee_batch_id)
# async def get_all_batch_oee(get_oee_request: GetOEERequestOneBatch,
#                             db: Session = Depends(get_db)):
#     try:
#         return await api_handler.get_oee_batch(get_oee_request=get_oee_request, db=db)
#     except Exception as e:
#         tb = traceback.format_exc()
#         logger.exception(e)
#         logger.exception(tb)
#         return DefaultFailureResponse(error=e.args)
