import time

from sqlalchemy.orm import Session

from scripts.core.engine.chart_creators import ChartMaker
from scripts.core.engine.oee_aggregator import processor,   aggregator
from scripts.db_layer import batch_oee_table
from scripts.logging.logging import logger
from scripts.schemas.batch_oee import (
    GetOEERequest,
    GetOEERequestOneBatch,
    GetBatches,
    GetProducts,
    ChartRequest,
)
from scripts.schemas.meta import LabelValue
from scripts.utils.mongo_util import MongoConnect


class APIHandler:
    @staticmethod
    async def get_oee_all(db: Session, get_oee_request: GetOEERequest):
        try:
            table_obj = batch_oee_table.BatchOEETable(db=db)
            data = table_obj.get_oee_data_all(
                start_time=get_oee_request.start_time,
                end_time=get_oee_request.end_time,
                hierarchy=get_oee_request.hierarchy,
            )
            return data
        except Exception:
            raise
        finally:
            del table_obj

    @staticmethod
    async def get_oee_batch(db: Session, get_oee_request: GetOEERequestOneBatch):
        try:
            table_obj = batch_oee_table.BatchOEETable(db=db)
            data = table_obj.get_oee_data_batch_id(
                hierarchy=get_oee_request.hierarchy, batch_id=get_oee_request.batch_id
            )
            return data
        except Exception:
            raise
        finally:
            del table_obj

    @staticmethod
    async def get_batches(db: Session, request_data: GetBatches):
        try:
            table_obj = batch_oee_table.BatchOEETable(db=db)
            if not request_data.end_time:
                request_data.end_time = int(time.time() * 1000)
            data = table_obj.get_batches(
                hierarchy=request_data.hierarchy,
                start_time=request_data.start_time,
                end_time=request_data.end_time,
            )
            return data
        except Exception:
            raise
        finally:
            del table_obj

    @staticmethod
    async def get_products(db: Session, request_data: GetProducts):
        try:
            table_obj = batch_oee_table.BatchOEETable(db=db)
            data = table_obj.get_products(
                hierarchy=request_data.hierarchy,
                start_time=request_data.queryDate[0],
                end_time=request_data.queryDate[1],
            )

            return [
                LabelValue(
                    label=each[0], value=each[0], start_time=each[1], end_time=each[2]
                ) if isinstance(each, list) else LabelValue(
                    label=each['batch_id'], value=each['batch_id'],
                    start_time=each['batch_start_time'], end_time=each['batch_end_time']
                )

                for each in data
            ]
        except Exception as e:
            logger.exception(e, exc_info=True)
            raise
        finally:
            del table_obj

    @staticmethod
    async def get_chart_data(db: Session, request_data: ChartRequest):
        try:
            table_obj = batch_oee_table.BatchOEETable(db=db)
            if not request_data.hierarchy:
                return dict()
            chart_maker = ChartMaker()
            data = table_obj.get_chart_data(
                hierarchy=request_data.hierarchy,
                start_time=request_data.queryDate[0],
                end_time=request_data.queryDate[1],
                product_id=request_data.productId,
                aggregation=request_data.aggregation,
            )
            if not request_data.aggregation or len(data) == 1:
                if isinstance(data, list):
                    data = data[0]
                raw_data = processor(data)
                return chart_maker.main_creator(raw_data, overall=False)
            elif len(data) == 0:
                return dict()
            else:
                agg_data = aggregator(data)
                return chart_maker.main_creator(agg_data)

        except Exception as e:
            raise
        finally:
            del table_obj
