import json
import time
from datetime import datetime

import pytz

from scripts.config import read_settings, DBConf
from scripts.core.engine.automation_engine import AutomationEngine
from scripts.db.mongo.dbs.siam_oee import SiamOEE
from scripts.db.redis_conn import live_tags_db_object


class ProductionMonitor:
    def __init__(self):
        self.automation_engine = AutomationEngine()
        self.settings = read_settings()
        self.oee_mongo = SiamOEE()
        self.machine_mode_tags = self.settings["automation"]["setup_time_logic"]["tags"]
        self.manual_mode_tag = self.settings["automation"]["setup_time_logic"]["manual_mode_tag"]
        self.auto_mode_tag = self.settings["automation"]["setup_time_logic"]["auto_mode_tag"]

        self.production_count_tags = self.settings["automation"]["production_end_logic"]["tags"]
        self.good_count_tag = self.settings["automation"]["production_end_logic"]["good_count_tag"]
        self.inspection_count_tag = self.settings["automation"]["production_end_logic"]["inspection_count_tag"]
        self.ng_count_tag = self.settings["automation"]["production_end_logic"]["ng_count_tag"]

    @staticmethod
    def get_redis_data(tag_list):
        tag_data = {}
        redis_response = live_tags_db_object.mget(tag_list)
        for index, each_tag in enumerate(tag_list):
            _val = redis_response.__getitem__(index)
            if not _val:
                continue
            _redis_resp = json.loads(_val)
            hierarchy = each_tag.removeprefix(DBConf.REDIS_PREFIX)
            if hierarchy not in tag_data:
                tag_data[hierarchy] = dict()
            tag_data[hierarchy] = _redis_resp.get("value")
        return tag_data

    def calculate_oee_params(self, data, downtime):
        start_time = datetime.fromtimestamp(
            data.get("start_time") // 1000, tz=pytz.timezone("Asia/Bangkok"))
        end_time = datetime.now(tz=pytz.timezone("Asia/Bangkok"))

        available_time = (end_time - start_time).total_seconds() / 60

        if downtime > available_time:
            downtime = 0

        operating_time = available_time - downtime

        availability = operating_time / available_time

        good_count, units_produced = self.get_current_produced_count()
        if not good_count:
            good_count = 0

        if not units_produced:
            units_produced = 0

        productive_time = units_produced * (1 / data.get("cycle_time"))

        performance = productive_time / operating_time

        if units_produced:
            quality = good_count / units_produced
        else:
            quality = 0

        oee = availability * performance * quality

        return oee * 100, availability * 100, performance * 100, quality * 100

    def calculate_setup_time(self, production_start_time: datetime, tz):
        tag_list = [f"{DBConf.REDIS_PREFIX}{i}" for i in self.machine_mode_tags]
        while True:
            try:
                if (datetime.now(tz=pytz.timezone(tz)) - production_start_time).total_seconds() > 600:
                    return round((datetime.now(tz=pytz.timezone(tz)) - production_start_time).total_seconds() / 60)

                tag_data = self.get_redis_data(tag_list)

                if tag_data.get(self.manual_mode_tag) == 0 and tag_data.get(self.auto_mode_tag) == 1:
                    print("production started!!!")
                    return round((datetime.now(tz=pytz.timezone(tz)) - production_start_time).total_seconds() / 60)
                print(tag_data)
                time.sleep(1)
            except Exception as e:
                print(e)

    def check_mongo_for_finish(self, job_id, machine_type) -> bool:
        data = self.oee_mongo.find_record(job_id, machine_type)
        if data.get("prod_status", "") == "completed":
            return True
        else:
            return False

    def check_production_end(self, data):
        total_count = data.get("actual_received_qty", data.get("qty_released"))
        if total_count <= 0:
            total_count = data.get("qty_released")
        job_id = data.get("job", "")
        machine_type = data.get("uf_process", "")
        tag_list = [f"{DBConf.REDIS_PREFIX}{i}" for i in self.production_count_tags]
        while True:
            if self.check_mongo_for_finish(job_id, machine_type):
                break
            try:
                tag_data = self.get_redis_data(tag_list)
                if tag_data.get(self.good_count_tag, 0) >= total_count \
                        or tag_data.get(self.inspection_count_tag, 0) >= total_count \
                        or tag_data.get(self.inspection_count_tag, -1) == 0 \
                        or tag_data.get(self.good_count_tag, -1) == 0:
                    print("production ended")
                    break
                time.sleep(1)
            except Exception as e:
                print(e)
        return True

    def check_production_run(self):
        tag_list = [f"{DBConf.REDIS_PREFIX}{i}" for i in self.production_count_tags]
        while True:
            try:
                tag_data = self.get_redis_data(tag_list)
                if tag_data.get(self.inspection_count_tag) > 0:
                    return True
                time.sleep(1)
            except Exception as e:
                print(e)

    def get_current_produced_count(self):
        tag_list = [f"{DBConf.REDIS_PREFIX}{i}" for i in self.production_count_tags]
        try:
            tag_data = self.get_redis_data(tag_list)
            return (
                tag_data.get(self.good_count_tag),
                tag_data.get(self.ng_count_tag, 0) + tag_data.get(self.good_count_tag, 0)
            )
        except Exception as e:
            print(e)
