import pytz

from scripts.errors import DataNotFound

if __name__ == '__main__':
    from dotenv import load_dotenv

    load_dotenv()
import os
import time
from datetime import datetime

from scripts.constants import CommonConstants, TagCategoryConstants
from scripts.constants.db_connections import mongo_client
from scripts.core.engine.oee_calculator import OEEEngine
from scripts.core.handlers.batch_oee_calc_handler import CalculateBatchOEEHandler
from scripts.core.handlers.common_handler import CommonHandler
from scripts.db.mongo.ilens_configuration.aggregations.customer_projects import ProjectAggregate
from scripts.db.mongo.ilens_configuration.collections.customer_projects import CustomerProjects
from scripts.logging import logger
from scripts.schemas.batch_oee import MachineOEERequest, BatchOEEData, OEEDataInsertRequest, OEEDataSaveRequest
from scripts.utils.common_utils import CommonUtils
from scripts.utils.kafka_util import DataPush

customer_conn = CustomerProjects(mongo_client=mongo_client)
customer_agg = ProjectAggregate()


class MachineOEECalculator:
    def __init__(self, project_id=None):
        self.common_util = CommonUtils()
        self.batch_oee_handler = CalculateBatchOEEHandler()
        self.common_handler = CommonHandler(project_id=project_id)
        self.oee_engine = OEEEngine()
        self.data_push = DataPush()

    def calculate_machine_oee(self, request_data: MachineOEERequest):
        try:
            hierarchy_dict = self.common_handler.get_valid_oee_monitoring_hierarchy(project_id=request_data.project_id)
            # now = datetime.today() - timedelta(days=1)
            now = datetime.today()
            start_timestamp = int(datetime.strptime(request_data.monitor_time, '%H:%M').replace(year=now.year,
                                                                                                month=now.month,
                                                                                                day=now.day).timestamp()) * 1000
            oee_start_time = datetime.strptime(request_data.monitor_time, '%H:%M').replace(year=now.year,
                                                                                           month=now.month,
                                                                                           day=now.day).strftime(
                CommonConstants.USER_META_TIME_FORMAT)
            curr_end_time = datetime.now().astimezone(tz=pytz.timezone(request_data.tz))
            curr_timestamp = int(curr_end_time.timestamp()) * 1000
            oee_end_time = curr_end_time.strftime(CommonConstants.USER_META_TIME_FORMAT)
            for k, v in hierarchy_dict.items():
                site_id = k.split("$")[0]
                if not v.get(TagCategoryConstants.OEE_CYCLE_DESIGN_CATEGORY):
                    logger.debug(f"Design Parameter(CYCLE TIME) not found for selected hierarchy {v}")
                    continue
                cycle_time_id = v[TagCategoryConstants.OEE_CYCLE_DESIGN_CATEGORY].split("$")[-1]
                cycle_time = self.common_handler.get_cycle_time_value_from_hierarchy(tag_id=cycle_time_id)
                if isinstance(cycle_time, bool) and not cycle_time:
                    logger.debug(f"OEE Cycle Design parameter details not found for selected hierarchy")
                    continue
                cal_type = self.common_util.get_uom_type(uom_type=os.environ.get("DEFAULT_UOM_TYPE", default="mins"))
                downtime = self.common_util.get_downtime_details_by_hierarchy(
                    hierarchy=k, project_id=request_data.project_id, uom_type=cal_type,
                    filters={"start_time": [start_timestamp, curr_timestamp]})
                if downtime is None:
                    logger.debug("Downtime for selected hierarchy got None, Updating to zero")
                    downtime = 0
                input_data = OEEDataInsertRequest(prod_start_time=oee_start_time,
                                                  prod_end_time=oee_end_time, downtime=downtime,
                                                  hierarchy=k, cycle_time=cycle_time,
                                                  tz=request_data.tz,
                                                  project_id=request_data.project_id)
                try:
                    input_data.total_units, input_data.reject_units = self.batch_oee_handler.get_data_for_tags(
                        input_data=input_data)
                    oee_response: BatchOEEData = self.oee_engine.start_batch_oee_calc(
                        request_data=OEEDataSaveRequest(**input_data.dict()))
                except DataNotFound:
                    logger.exception(f"Data Not Found for selected Hierarchy --- {k}")
                    continue
                except Exception as e:
                    logger.exception(
                        f"Exception Occurred while calculating oee {e.args}, skipping oee calculation for hierarchy - {k} ")
                    continue
                data_dict = {
                    v[TagCategoryConstants.OEE_OUTPUT_TOTAL_UNITS_CATEGORY]: oee_response.total_units,
                    v[TagCategoryConstants.OEE_OUTPUT_REJECT_UNITS_CATEGORY]: oee_response.reject_units,
                    v[TagCategoryConstants.OEE_OUTPUT_CATEGORY]: oee_response.oee,
                    v[TagCategoryConstants.OEE_OUTPUT_PERFORMANCE_CATEGORY]: oee_response.performance,
                    v[TagCategoryConstants.OEE_OUTPUT_QUALITY_CATEGORY]: oee_response.quality,
                    v[TagCategoryConstants.OEE_OUTPUT_QUALITY_LOSS_CATEGORY]: oee_response.quality_loss,
                    v[TagCategoryConstants.OEE_OUTPUT_PERFORMANCE_LOSS_CATEGORY]: oee_response.performance_loss,
                    v[TagCategoryConstants.OEE_OUTPUT_AVAILABILITY_CATEGORY]: oee_response.availability,
                    v[TagCategoryConstants.OEE_OUTPUT_AVAILABILITY_LOSS_CATEGORY]: oee_response.availability_loss,
                    v[TagCategoryConstants.OEE_OUTPUT_DOWNTIME_CATEGORY]: oee_response.downtime
                }
                message_dict = {
                    "data": data_dict,
                    "site_id": site_id,
                    "gw_id": "",
                    "pd_id": "",
                    "p_id": request_data.project_id,
                    "timestamp": curr_timestamp,
                    "msg_id": 1,
                    "retain_flag": False
                }
                self.data_push.publish_message(message_dict)

        except Exception as e:
            logger.exception(f"Exception Occurred while calculating oee for the hierarchy {e.args}")
        return


if __name__ == '__main__':
    while True:
        project_dict = customer_conn.find_project_by_aggregate(customer_agg.get_tz_mapping_query_with_project_id())
        project_dict = project_dict[0] if project_dict else {}
        projects_list = os.environ.get("OEE_PROJECTS", default="project_170")
        monitor_start_time = os.environ.get("OEE_START_TIME", default="00:00")
        for project in projects_list.split(","):
            MachineOEECalculator().calculate_machine_oee(
                request_data=MachineOEERequest(project_id=project, monitor_time="00:00",
                                               tz=project_dict.get(project, "Asia/Kolkata")))
        sleep_time_in_mins = int(os.environ.get("AUTOMATION_SLEEP_TIME_IN_MINS", default=1))
        sleep_time_in_seconds = sleep_time_in_mins * 60
        time.sleep(sleep_time_in_seconds)
