from datetime import datetime

import pandas as pd
import pytz
from sqlalchemy import create_engine

from scripts.config import DBConf
from scripts.config import Metadata
from scripts.core.engine.automation_engine import AutomationEngine
from scripts.core.handlers.batch_oee_calc_handler import CalculateBatchOEEHandler, OEEDataInsertRequest
from scripts.db.mongo.dbs.siam_oee import SiamOEE
from scripts.db_layer.job_table import JobTable
from scripts.logging.logging import logger as logging
from scripts.models.db_models import JobTable as JobSkeleton
from scripts.schemas.form import (GetRDBValues, CustomQuery, StartProductionRequest, EndProductionRequest,
                                  StartProdJobModel, EndProdJobModel, CalculateOEE, FormSaveRequest, FormDetails,
                                  EndProdJobDB)
from scripts.utils.common_utils import CommonUtils


class FormHandler:

    def __init__(self):
        self.oee_mongo = SiamOEE()
        self.automation_engine = AutomationEngine()

    @staticmethod
    async def fetch_last_values(request_data: GetRDBValues):
        try:
            base_engine = create_engine(f"{DBConf.CLIENT_URI}{request_data.db_name}")
            query = f"SELECT * from {request_data.table_name}"
            if request_data.primary_conditions:
                query += " WHERE "
                for column, val in request_data.primary_conditions.items():
                    query += f"{column}='{val}'"
            query += " LIMIT 1"
            table_data_df = pd.read_sql(query, base_engine)
            del base_engine
            table_data_df.rename(columns=request_data.column_to_property, inplace=True)
            return table_data_df.to_dict(orient="records")[0]
        except Exception as e:
            logging.exception(e)

    @staticmethod
    async def custom_query_fetch(request_data: CustomQuery):
        try:
            base_engine = create_engine(f"{DBConf.CLIENT_URI}{request_data.db_name}")
            table_data_df = pd.read_sql(request_data.query, base_engine)
            del base_engine
            table_data_df.rename(columns=request_data.column_to_property, inplace=True)

            automation_eng = AutomationEngine()
            current_time = datetime.now()
            abs_start_time = automation_eng.get_absolute_start_time(button_click_time=current_time)

            erp_table_data = table_data_df.to_dict(orient="records")[0]
            erp_table_data.update({
                "start_time": abs_start_time.strftime("%Y-%m-%d %H:%M")
            })
            return erp_table_data
        except Exception as e:
            logging.exception(e)

    async def start_production(self, request_data: StartProductionRequest, db_session):
        try:
            table_data = JobTable(db_session)
            request_data.submitted_data.update(tz=request_data.tz)
            job_model = StartProdJobModel(**request_data.submitted_data.get("data", {}))
            job_data = job_model.dict(exclude_none=True)
            job_data.pop("tz", None)
            row_data = JobSkeleton(**job_data)
            table_data.add_data(row_data)

            # TODO: Create mongo record with job details
            job_data.update(form_details=FormDetails(**request_data.dict()).dict(),
                            prod_status="started")
            self.oee_mongo.update_oee(job_data, job_model.job, job_model.uf_process)
        except Exception as e:
            logging.exception(e)

    async def end_production(self, request_data: EndProductionRequest, db_session, request_cookies):
        try:
            table_data = JobTable(db_session)
            job_model, db_data = await self.get_job_data(request_data)
            table_data.update_record(job_model.job, job_model.uf_process, db_data.dict(exclude_none=True))
            calculate_oee_payload = CalculateOEE(batch_start_time=job_model.prod_start_time,
                                                 batch_end_time=job_model.prod_end_time,
                                                 batch_id=job_model.job,
                                                 setup_time=job_model.setup_time,
                                                 cycle_time=job_model.cycle_time,
                                                 total_units=job_model.qty_released)
            calculate_oee_payload.downtime = await self.get_oee_downtime(request_data.submitted_data["data"],
                                                                         job_model.prod_end_time, job_model.tz)
            _ = await CalculateBatchOEEHandler().calculate_oee(db_session, OEEDataInsertRequest(
                **calculate_oee_payload.dict()))
            form_response = await self.save_to_form(request_data, request_cookies, job_model)
            logging.info(f"FORM SAVE RESPONSE, {form_response}")
            return "Form values updated successfully"
        except Exception as e:
            logging.exception(e)
            return f"Server encountered an error during op: {e}"

    async def get_job_data(self, request_data: EndProductionRequest):
        request_data.submitted_data.update(tz=request_data.tz)
        form_data = request_data.submitted_data.get("data", {})
        job = form_data.get("job")
        uf_process = form_data.get("uf_process")
        data_from_mongo = self.oee_mongo.find_record(job, uf_process)
        form_data.update(data_from_mongo)
        job_model = EndProdJobModel(**form_data)
        if not job_model.setup_time:
            job_model.setup_time = 0
        if data_from_mongo.get("units_produced"):
            job_model.qty_released = data_from_mongo.get("units_produced")
        db_data = EndProdJobDB(**job_model.dict())
        return job_model, db_data

    async def save_to_form(self, request_data: EndProductionRequest,
                           request_cookies,
                           job_model: EndProdJobModel):
        end_date_time = datetime.fromtimestamp(job_model.prod_end_time // 1000, tz=pytz.timezone(request_data.tz))
        end_str = end_date_time.strftime("%Y-%m-%d %H:%M")

        start_date_time = datetime.fromtimestamp(job_model.prod_start_time // 1000, tz=pytz.timezone(request_data.tz))
        start_str = start_date_time.strftime("%Y-%m-%d %H:%M")

        tag_data = self.automation_engine.get_all_tags(end_date_time)
        tag_data.update(**job_model.dict(exclude_none=True))
        form_save_payload = FormSaveRequest(**request_data.dict())
        form_save_payload.submitted_data["data"].update(**tag_data)
        form_save_payload.submitted_data["data"].update(end_time=end_str, start_time=start_str)

        form_response = await CommonUtils.hit_external_service(api_url=f"{Metadata.FORM_API}render/form?save=True",
                                                               payload=form_save_payload.dict(),
                                                               request_cookies=request_cookies)
        form_save_payload.submitted_data["data"].update({"prod_status": "completed"})
        self.oee_mongo.update_oee(form_save_payload.submitted_data["data"], job_model.job, job_model.uf_process)
        return form_response

    async def get_oee_downtime(self, data, end_time, tz):
        if isinstance(end_time, int):
            end_time = datetime.fromtimestamp(end_time // 1000, tz=pytz.timezone(tz))
        run_start_time = data.get("start_time") if not data.get("run_start_time") else data.get("run_start_time")

        if isinstance(run_start_time, int):
            run_start_time = datetime.fromtimestamp(run_start_time // 1000, tz=pytz.timezone(tz))
        else:
            run_start_time = datetime.strptime(run_start_time, "%Y-%m-%d %H:%M")
            run_start_time = run_start_time.replace(tzinfo=pytz.timezone(tz))
        return self.automation_engine.get_downtime(run_start_time, end_time)
