import time

from scripts.errors import ErrorCodes

from scripts.constants import UOM
from scripts.logging import logger
from scripts.schemas.batch_oee import BatchOEEDataSaveRequest, BatchOEEData


class OEECalculator:
    @staticmethod
    async def calculate_availability(operating_time, planned_prod_time):
        if operating_time > planned_prod_time:
            logger.error(ErrorCodes.ERR001)
            raise ValueError(ErrorCodes.ERR001)
        try:
            return operating_time / planned_prod_time
        except Exception as e:
            logger.exception(e)
            raise

    @staticmethod
    async def calculate_performance(units_produced, cycle_time, operating_time):
        try:
            if cycle_time == 0 or operating_time == 0:
                logger.error(ErrorCodes.ERR002)
                raise ValueError(ErrorCodes.ERR002)
            productive_time = units_produced * (1 / cycle_time)
            if productive_time > operating_time:
                logger.error(ErrorCodes.ERR003)
                raise ValueError(ErrorCodes.ERR003)
            return productive_time / operating_time
        except Exception as e:
            logger.exception(e)
            raise

    @staticmethod
    async def calculate_productive_time(units_produced, cycle_time):
        try:
            if cycle_time == 0:
                logger.error(ErrorCodes.ERR002)
                raise ValueError(ErrorCodes.ERR002)
            return units_produced * (1 / cycle_time)

        except Exception as e:
            logger.exception(e)
            raise

    @staticmethod
    async def calculate_quality(rejected_units, total_units):
        if rejected_units > total_units:
            logger.error(ErrorCodes.ERR004)
            raise ValueError(ErrorCodes.ERR004)
        try:
            return (total_units - rejected_units) / total_units
        except ZeroDivisionError:
            return 0
        except Exception as e:
            logger.exception(e)
            raise

    @staticmethod
    async def calculate_oee(availability, performance, quality):
        try:
            return availability * performance * quality
        except Exception as e:
            logger.exception(e)
            raise


class OEELossesCalculator:
    @staticmethod
    async def calculate_availability_loss(downtime, available_time):
        return (downtime / available_time) * 100

    @staticmethod
    async def calculate_quality_loss(reject_units, cycle_time, available_time):
        return ((reject_units * (1 / cycle_time)) / available_time) * 100

    @staticmethod
    async def calculate_performance_loss(
        oee_percentage, availability_loss, quality_loss
    ):
        return 100 - availability_loss - quality_loss - oee_percentage


class OEEEngine:
    def __init__(self):
        self.oee_calc = OEECalculator()
        self.oee_loss_calc = OEELossesCalculator()

    async def start_batch_oee_calc(
            self,
            product_info: BatchOEEDataSaveRequest
    ) -> BatchOEEData:
        try:
            logger.debug(f"Calculating OEE for {product_info.batch_id}")

            # Start and End time should be in milliseconds since epoch.
            if product_info.uom == UOM.minutes:
                divisor = UOM.time_divs.minutes
            elif product_info.uom == UOM.seconds:
                divisor = UOM.time_divs.seconds
            elif product_info.uom == UOM.hours:
                divisor = UOM.time_divs.hours
            elif product_info.uom == UOM.millis:
                divisor = UOM.time_divs.millis
            else:
                divisor = UOM.time_divs.minutes

            planned_production_time = (
                product_info.batch_end_time - product_info.batch_start_time
            ) / divisor

            operating_time = planned_production_time - product_info.downtime

            availability = await self.oee_calc.calculate_availability(
                operating_time=operating_time,
                planned_prod_time=planned_production_time,
            )

            performance = await self.oee_calc.calculate_performance(
                units_produced=product_info.total_units,
                operating_time=operating_time,
                cycle_time=product_info.cycle_time,
            )

            quality = await self.oee_calc.calculate_quality(
                total_units=product_info.total_units,
                rejected_units=product_info.reject_units,
            )

            oee = await self.oee_calc.calculate_oee(
                availability=availability,
                performance=performance,
                quality=quality,
            )

            productive_time = await self.oee_calc.calculate_productive_time(
                cycle_time=product_info.cycle_time,
                units_produced=product_info.total_units,
            )

            availability_loss = await self.oee_loss_calc.calculate_availability_loss(
                downtime=product_info.downtime,
                available_time=planned_production_time,
            )

            quality_loss = await self.oee_loss_calc.calculate_quality_loss(
                reject_units=product_info.reject_units,
                available_time=planned_production_time,
                cycle_time=product_info.cycle_time,
            )

            performance_loss = await self.oee_loss_calc.calculate_performance_loss(
                oee_percentage=oee * 100,
                availability_loss=availability_loss,
                quality_loss=quality_loss,
            )

            oee_dict = {
                "availability": availability * 100,
                "performance": performance * 100,
                "quality": quality * 100,
            }

            oee_loss = {
                "availability_loss": availability_loss,
                "quality_loss": quality_loss,
                "performance_loss": performance_loss,
            }

            logger.debug(f"OEE: {product_info.batch_id}: {oee_dict}")
            logger.debug(f"OEE Loss: {product_info.batch_id}: {oee_loss}")

            batch_oee = BatchOEEData(
                **product_info.dict(),
                calculated_on=int(time.time() * 1000),
                productive_time=productive_time,
                availability=availability * 100,
                performance=performance * 100,
                quality=quality * 100,
                availability_loss=availability_loss,
                quality_loss=quality_loss,
                performance_loss=performance_loss,
                oee=oee * 100,
            )

            return batch_oee
        except Exception:
            raise
