from abc import abstractmethod
from urllib.parse import urlparse

import aiohttp
import httpx
import requests
from aiohttp import ClientResponse

from scripts.constants import CommonStatusCode, Secrets
from scripts.logging import logger
from scripts.utils.security_utils.apply_encrytion_util import create_token


class AuthenticationError(Exception): ...


class ForbiddenError(Exception): ...


token = ""
secrets = {}


class BaseRequestHandler:
    """
    BaseRequestHandler
    """

    def __init__(self, url, time_out=None) -> None:
        self.time_out = time_out
        self.url = url
        self.verify = False

    @property
    def get_timeout(self):
        return self.time_out

    def post(self, path="", json=None, data=None, update_args=True, **kwargs) -> requests.Response:
        """doc"""
        url = self.get_url(path)

        logger.debug(f"url path -- {url}")
        response = requests.post(url=url, data=data, json=json, **self.prepare_args(**kwargs))

        if response.status_code not in CommonStatusCode.SUCCESS_CODES:
            logger.exception(f"status - {response.status_code}")
            raise AuthenticationError(f"Connection Failure.. Status Code - {response.status_code}")

        if update_args:
            self.update_args(response=response)
        return response

    def get(self, path="", params=None, **kwargs) -> requests.Response:
        """doc"""
        url = self.get_url(path)
        # Commenting this as currently there is no use case for this
        # self.update_args(response = response)
        response = requests.get(url=url, params=params, **self.prepare_args(**kwargs))
        if response.status_code == 401:
            raise AuthenticationError("Not authorized")

        if response.status_code == 403:
            raise ForbiddenError("Not Permitted")
        return response

    def get_url(self, path):
        return f"{self.url.rstrip('/')}/{path.lstrip('/').rstrip('/')}"

    def verify_request(self):
        if self.url_scheme(self.url) == 'https':
            self.verify = True
        return self.verify

    @staticmethod
    def url_scheme(url):
        return urlparse(url).scheme

    @abstractmethod
    def prepare_args(self, **kwargs):
        ...

    @abstractmethod
    def update_args(self, **kwargs):
        ...


class HTTPXRequestHandler:
    def __init__(self, url, time_out=None) -> None:
        self.time_out = time_out
        self.url = url
        self.verify = False

    @property
    def get_timeout(self):
        return self.time_out

    def httpx_post(self, path="", json=None, data=None, update_args=True, **kwargs) -> requests.Response:
        """doc"""
        url = self.get_url(path)
        print(url)
        with httpx.Client() as client:
            response = client.post(url=url, data=data, json=json, **self.prepare_args(**kwargs))

        if response.status_code == 401:
            raise AuthenticationError("Not authorized")

        if response.status_code == 403:
            raise ForbiddenError("Not Permitted")

        if update_args:
            self.update_args(response=response)
        return response

    def httpx_get(self, path="", params=None, **kwargs) -> requests.Response:
        """doc"""
        url = self.get_url(path)
        # Commenting this as currently there is no use case for this
        # self.update_args(response = response)
        with httpx.Client() as client:
            response = client.get(url=url, params=params, **self.prepare_args(**kwargs))

        if response.status_code == 401:
            raise AuthenticationError("Not authorized")

        if response.status_code == 403:
            raise ForbiddenError("Not Permitted")
        return response

    def get_url(self, path):
        return f"{self.url.rstrip('/')}/{path.lstrip('/').rstrip('/')}"

    def verify_request(self):
        if self.url_scheme(self.url) == 'https':
            self.verify = True
        return self.verify

    @staticmethod
    def url_scheme(url):
        return urlparse(url).scheme

    @abstractmethod
    def prepare_args(self, **kwargs):
        ...

    @abstractmethod
    def update_args(self, **kwargs):
        ...


class AIOHTTPRequestHandler:
    def __init__(self, url, time_out=None) -> None:
        self.time_out = time_out
        self.url = url
        self.verify = False

    @property
    def get_timeout(self):
        return self.time_out

    async def aiohttp_post(self, path="", json=None, data=None, update_args=True, **kwargs) -> ClientResponse:
        """doc"""
        url = self.get_url(path)
        print(url)
        async with aiohttp.ClientSession() as client:
            async with client.post(
                    url=url, data=data, json=json, **self.prepare_args(**kwargs)
            ) as resp:
                response = resp

        if response.status == 401:
            raise AuthenticationError("Not authorized")

        if response.status == 403:
            raise ForbiddenError("Not Permitted")

        if update_args:
            self.update_args(response=response)
        return response

    async def aiohttp_get(self, path="", params=None, **kwargs) -> ClientResponse:
        """doc"""
        url = self.get_url(path)
        # Commenting this as currently there is no use case for this
        # self.update_args(response = response)
        async with aiohttp.ClientSession() as client:
            async with client.get(
                    url=url, params=params, **self.prepare_args(**kwargs)
            ) as resp:
                response = resp

        if response.status == 401:
            raise AuthenticationError("Not authorized")

        if response.status == 403:
            raise ForbiddenError("Not Permitted")
        return response

    def get_url(self, path):
        return f"{self.url.rstrip('/')}/{path.lstrip('/').rstrip('/')}"

    def verify_request(self):
        if self.url_scheme(self.url) == 'https':
            self.verify = True
        return self.verify

    @staticmethod
    def url_scheme(url):
        return urlparse(url).scheme

    @abstractmethod
    def prepare_args(self, **kwargs):
        ...

    @abstractmethod
    def update_args(self, **kwargs):
        ...


class ILensRequest(BaseRequestHandler, HTTPXRequestHandler, AIOHTTPRequestHandler):
    """
    Utility to use ilens API's Directly
    """

    def __init__(self, url, project_id) -> None:
        super().__init__(url)
        self.project_id = project_id

    def prepare_args(self, **kwargs) -> dict:
        """
        doc
        """
        post_args = {}
        post_args.update(**kwargs)
        post_args.update(headers=self.get_headers(**kwargs))
        post_args.update(timeout=self.get_timeout)
        post_args.update(verify=self.verify_request())
        post_args.update(cookies=self.get_cookies(**kwargs))
        return post_args

    def update_args(self, **kwargs) -> bool:
        data = kwargs.get('response').headers.get('token')
        if data:
            global token
            token = data
        return data

    def get_headers(self, **kwargs) -> dict:
        headers = {'X-Content-Type-Options': 'nosniff',
                   'X-Frame-Options': 'SAMEORIGIN', 'Cache-Control': 'no-store', 'projectId': self.project_id}
        headers.update(kwargs.get("headers", {}))
        return headers

    @staticmethod
    def get_cookies(**kwargs) -> dict:
        cookies = {}
        cookies.update(kwargs.get("cookies", {}))
        return cookies

    @staticmethod
    def create_token(host: str = '127.0.0.1', user_id=None, internal_token=Secrets.token):
        """
        This method is to create a cookie
        """

        try:
            if user_id is None:
                user_id = "user_099"
            new_token = create_token(
                user_id=user_id,
                ip=host,
                token=internal_token,
            )
            return new_token
        except Exception as e:
            logger.exception(str(e))
            raise
