import pandas as pd

from scripts.schemas.batch_oee import ChartResponse, ChartDBResponse
from scripts.utils.common_utils import CommonUtils


class OEEAggregator:
    def __init__(self):
        self.common_util = CommonUtils()

    def processor(self, data):
        db_response = ChartDBResponse(**data)
        duration = self.common_util.get_duration(meta=data, difference=True, tz=data["tz"])
        cal_type = self.common_util.get_uom_type(uom_type=data["uom"])
        db_response.total_time = self.common_util.get_diff_duration_in_int(input_time=duration, return_type=cal_type)
        db_response.actual_cycle = round(
            db_response.total_units / db_response.total_time, 2
        )
        db_response.ideal_cycle = round(db_response.cycle_time, 2)
        db_response.good_units = round(
            db_response.total_units - db_response.reject_units, 2
        )

        chart_response = ChartResponse(**db_response.dict())
        return chart_response.dict()

    @staticmethod
    def aggregator(data, activity_length=1):
        df = pd.DataFrame(data)
        df["total_time"] = (df["batch_end_time"] - df["batch_start_time"]) / 60000
        df["actual_cycle"] = df["total_units"] / df["total_time"]
        df["ideal_cycle"] = df["cycle_time"]
        df["good_units"] = df["total_units"] - df["reject_units"]
        df["reject_time"] = df["reject_units"] * (1 / df["ideal_cycle"])
        agg_oee = df.sum().round(2)
        availability = (agg_oee["total_time"] - agg_oee["downtime"]) / agg_oee["total_time"]
        performance = agg_oee["productive_time"] / (
                agg_oee["total_time"] - agg_oee["downtime"]
        )
        quality = (agg_oee["total_units"] - agg_oee["reject_units"]) / agg_oee[
            "total_units"
        ]
        oee_overall = round(availability * performance * quality, 2) * 100
        availability_loss = agg_oee["downtime"] / agg_oee["total_time"] * 100
        quality_loss = agg_oee["reject_time"] / agg_oee["total_time"] * 100
        chart_response = ChartResponse(
            total_units=round(agg_oee["total_units"] - (len(df) * activity_length)),
            reject_units=agg_oee["reject_units"],
            oee=oee_overall,
            availability=round(availability * 100, 2),
            downtime=agg_oee["downtime"],
            performance=round(performance * 100, 2),
            quality=round(quality * 100, 2),
            actual_cycle=agg_oee["actual_cycle"],
            ideal_cycle=agg_oee["ideal_cycle"],
            good_units=round(agg_oee["good_units"] - (len(df) * activity_length)),
            availability_loss=availability_loss,
            quality_loss=quality_loss,
            performance_loss=round(100 - availability_loss - quality_loss - oee_overall, 2),
            total_time=agg_oee["total_time"],
            productive_time=agg_oee["productive_time"],
        )
        filtered = chart_response.dict()
        remove_keys = ["productive_time", "downtime", "reject_units"]
        [filtered.pop(each, None) for each in remove_keys]
        return filtered
