import logging
from typing import List

import pandas as pd

from scripts.config import DBConf
from scripts.errors import DataFrameFormationError, ErrorMessages
from . import KairosConstants


def get_group_by(group_by_dict, group_by_tag_list, grouping_order):
    hierarchy_id = []
    if DBConf.KAIROS_DEFAULT_FULL_TAG in group_by_tag_list:
        return default_group_by(group_by_dict)
    for tag in grouping_order:
        for each_group in group_by_dict:
            if each_group.get("name") != "tag":  # Only group by tag is supported;
                continue
            all_groups = each_group.get("group", {})
            if tag in group_by_tag_list and tag in all_groups:
                hierarchy_id.append(all_groups[tag])
    if not hierarchy_id:
        raise DataFrameFormationError(ErrorMessages.DF_ERROR2)
    return "$".join(hierarchy_id)


def default_group_by(group_by_dict):
    for each_group in group_by_dict:
        if each_group.get("name") != "tag":  # Only group by tag is supported;
            continue
        all_groups = each_group.get("group", {})
        if DBConf.KAIROS_DEFAULT_FULL_TAG in all_groups:
            return all_groups[DBConf.KAIROS_DEFAULT_FULL_TAG]
    raise DataFrameFormationError(ErrorMessages.DF_ERROR2)


def form_df(metric, tags, values, df_index, tz, group_by_tags, grouping_order, group_by):
    """Helper function for parsing Kairos data and forming dataframe for each results"""
    df = pd.DataFrame(columns=["timestamp", "value"])
    try:
        selected_hierarchy = get_group_by(group_by, group_by_tags, grouping_order)
        if not tags or not values:
            new_column_name = selected_hierarchy
            df.rename(columns={"value": new_column_name}, inplace=True)
            df.set_index(df_index, inplace=True)
            return df
        df = pd.DataFrame(values, columns=KairosConstants().kairos_column_names)
        if KairosConstants().kairos_std_metric_name in metric:
            new_column_name = selected_hierarchy
            df.rename(columns={"value": new_column_name}, inplace=True)
        timezone = tz
        df[df_index] = pd.to_datetime(df[df_index], unit=KairosConstants.kairos_time_unit)
        df[df_index] = df[df_index].dt.tz_localize('UTC').dt.tz_convert(timezone)
        df.set_index(df_index, inplace=True)
        df.sort_index(inplace=True)
        df[f'{selected_hierarchy}_diff'] = df[selected_hierarchy].diff()
    except Exception as e:
        logging.exception(e)
    return df


def get_statistics(df, metadata):
    try:
        if metadata.enable_stats:
            metadata.data_statistics = {
                "min": df.min().round(2).to_dict(),
                "max": df.max().round(2).to_dict(),
                "avg": df.mean().round(2).to_dict()
            }
    except Exception as e:
        logging.exception(e)


def create_kairos_df(response_data, tags_list: List, group_by_tags: List, master_df=None, df_index='timestamp',
                     tz='Asia/kolkata'):
    """
    Definition for creating kairos DataFrame
    """
    if master_df is None:
        master_df = pd.DataFrame()

    try:
        for each_response in response_data.get("queries"):
            if each_response["sample_size"] == 0:
                continue
            for each_result in each_response["results"]:
                df = form_df(
                    metric=each_result.get("name"),
                    tags=each_result.get("tags"),
                    values=each_result.get("values"),
                    group_by=each_result.get("group_by"),
                    df_index=df_index,
                    tz=tz,
                    group_by_tags=group_by_tags,
                    grouping_order=tags_list
                )
                df = df[~df.index.duplicated()]
                master_df = master_df.combine_first(df)

        master_df = master_df.fillna(0)
        return master_df
    except Exception as e:
        raise DataFrameFormationError(e)
