import os

from scripts.constants import TagCategoryConstants
from scripts.constants.db_connections import mongo_client
from scripts.core.handlers.tag_handler import TagHierarchyHandler
from scripts.db.mongo.ilens_configuration.collections.lookup_table import LookupTable
from scripts.db.mongo.schema.tag_hierarchy import OutputTagsList
from scripts.logging import logger


class CommonHandler:
    def __init__(self, project_id=None):
        self.lookup_table = LookupTable(project_id=project_id, mongo_client=mongo_client)
        self.tag_hierarchy_handler = TagHierarchyHandler(project_id=project_id)

    def fetch_oee_hierarchy_from_look_up(self, project_id):
        oee_lookup_name = os.environ.get("OEE_LOOKUP_NAME", default="oee_monitoring")
        try:
            lookup_data = self.lookup_table.map_lookup_keys(lookup_name=oee_lookup_name, project_id=project_id)
            if not lookup_data:
                logger.debug(f"Lookup details not found for OEE")
                return {}
            return lookup_data
        except Exception as e:
            logger.exception(f"Failed to fetch hierarchy details from lookup {e.args}")
        return {}

    def get_valid_oee_monitoring_hierarchy(self, project_id):
        valid_hierarchies_dict = {}
        try:
            oee_lookup_dict = self.fetch_oee_hierarchy_from_look_up(project_id=project_id)
            hierarchy_list = list(oee_lookup_dict.values())
            if not hierarchy_list:
                logger.debug(f'Hierarchy details not found for the project {project_id} for OEE Monitoring!!!!')
                return {}
            tags_dict_by_hierarchy = self.tag_hierarchy_handler.get_output_tags_for_oee(
                OutputTagsList(project_id=project_id, hierarchy_list=hierarchy_list))
            for each_hierarchy in tags_dict_by_hierarchy:
                if not tags_dict_by_hierarchy.get(each_hierarchy):
                    logger.debug(f'Tag details not found for the hierarchy {each_hierarchy} for OEE Monitoring!!!!')
                    continue
                updated_dict = self.validate_hierarchy_tags(tags_dict_by_hierarchy[each_hierarchy])
                if not updated_dict:
                    continue
                valid_hierarchies_dict.update({each_hierarchy: updated_dict})
            return valid_hierarchies_dict
        except Exception as e:
            logger.exception(f"Exception Occurred while fetching hierarchy details for monitoring oee {e.args}")
        return {}

    @staticmethod
    def validate_hierarchy_tags(request_dict: dict):
        hierarchy_dict = {}
        category_constants_list = [v for k, v in TagCategoryConstants.__dict__.items() if not k.startswith("__")]
        try:
            for _each_tag in category_constants_list:
                if _each_tag in {TagCategoryConstants.TOTAL_UNITS_CATEGORY, TagCategoryConstants.REJECT_UNITS_CATEGORY}:
                    continue
                if not request_dict.get(_each_tag):
                    logger.debug(f"OEE Output Tag not configured for the hierarchy{list(request_dict.keys())[0]}")
                    return {}
                hierarchy_dict.update({_each_tag: request_dict[_each_tag]})
            return hierarchy_dict
        except Exception as e:
            logger.exception(f'Exception Occurred while validating the tags for a hierarchy {e.args}')
        return {}

    @staticmethod
    def get_oee_keys_mapping_dict(input_dict: dict):
        final_dict = {}
        for k, v in input_dict.items():
            if k == TagCategoryConstants.TOTAL_UNITS_CATEGORY:
                final_dict.update({v: "total_units"})
            elif k == TagCategoryConstants.REJECT_UNITS_CATEGORY:
                final_dict.update({v: "reject_units"})
            elif k == TagCategoryConstants.OEE_OUTPUT_CATEGORY:
                final_dict.update({v: "oee"})
            elif k == TagCategoryConstants.OEE_OUTPUT_AVAILABILITY_CATEGORY:
                final_dict.update({v: "availability"})
            elif k == TagCategoryConstants.OEE_OUTPUT_PERFORMANCE_CATEGORY:
                final_dict.update({v: "performance"})
            elif k == TagCategoryConstants.OEE_OUTPUT_QUALITY_CATEGORY:
                final_dict.update({v: "quality"})
            elif k == TagCategoryConstants.OEE_OUTPUT_QUALITY_LOSS_CATEGORY:
                final_dict.update({v: "quality_loss"})
            elif k == TagCategoryConstants.OEE_OUTPUT_AVAILABILITY_LOSS_CATEGORY:
                final_dict.update({v: "availability_loss"})
        return final_dict
