import time
from datetime import datetime

import pytz
from sqlalchemy.orm import Session

from scripts.core.engine.chart_creators import ChartMaker
from scripts.core.engine.oee_aggregator import OEEAggregator
from scripts.db.psql.oee_discrete import DiscreteOEE
from scripts.logging import logger
from scripts.schemas.batch_oee import (
    GetOEERequest,
    GetOEERequestOneBatch,
    GetBatches,
    BatchesGet,
    ChartRequest,
)
from scripts.schemas.meta import LabelValue


class APIHandler:

    def __init__(self):
        self.oee_agg = OEEAggregator()

    @staticmethod
    def get_oee_all(db: Session, get_oee_request: GetOEERequest):
        table_obj = DiscreteOEE(db=db)
        try:
            data = table_obj.get_oee_data_all(
                prod_start_time=get_oee_request.prod_start_time,
                prod_end_time=get_oee_request.prod_end_time,
                hierarchy=get_oee_request.hierarchy,
            )
            return data
        except Exception:
            raise

    @staticmethod
    def get_oee_batch(db: Session, get_oee_request: GetOEERequestOneBatch):
        table_obj = DiscreteOEE(db=db)
        try:
            data = table_obj.get_oee_data_by_reference_id(
                hierarchy=get_oee_request.hierarchy, reference_id=get_oee_request.reference_id,
                project_id=get_oee_request.project_id
            )
            return data
        except Exception:
            raise

    @staticmethod
    def get_batches(db: Session, request_data: GetBatches):
        table_obj = DiscreteOEE(db=db)
        try:
            if not request_data.prod_end_time:
                request_data.prod_end_time = int(time.time() * 1000)
            data = table_obj.get_batches(
                hierarchy=request_data.hierarchy,
                prod_start_time=request_data.prod_start_time,
                prod_end_time=request_data.prod_end_time,
            )
            return data
        except Exception:
            raise

    @staticmethod
    async def get_batches_info(db: Session, request_data: BatchesGet):
        table_obj = DiscreteOEE(db=db)
        try:
            data = table_obj.get_batches_info(
                hierarchy=request_data.hierarchy,
                prod_start_time=datetime.fromtimestamp(request_data.queryDate[0] / 1000).astimezone(
                    tz=pytz.timezone(request_data.tz)).isoformat(),
                prod_end_time=datetime.fromtimestamp(request_data.queryDate[-1] / 1000).astimezone(
                    tz=pytz.timezone(request_data.tz)).isoformat(), tz=request_data.tz
            )

            return [
                LabelValue(
                    label=each[0], value=each[0], start_time=each[1], end_time=each[2]
                ) if isinstance(each, list) else LabelValue(
                    label=each['reference_id'], value=each['reference_id'],
                    start_time=each['prod_start_time'], end_time=each['prod_end_time']
                )

                for each in data
            ]
        except Exception as e:
            logger.exception(e, exc_info=True)
            raise

    async def get_chart_data(self, db: Session, request_data: ChartRequest):
        table_obj = DiscreteOEE(db=db)
        try:
            if not request_data.hierarchy:
                return dict()
            chart_maker = ChartMaker()
            if request_data.reference_id:
                data = table_obj.get_chart_data(
                    hierarchy=request_data.hierarchy,
                    prod_start_time=request_data.queryDate[0],
                    prod_end_time=request_data.queryDate[1],
                    reference_id=request_data.reference_id
                )
                raw_data = self.oee_agg.processor(data)
                return chart_maker.main_creator(raw_data, overall=False)
            else:
                agg_data = self.oee_agg.aggregator(request_data=request_data)
                return chart_maker.main_creator(agg_data)

        except Exception as e:
            raise
        finally:
            del table_obj
