from copy import deepcopy

import pandas as pd

from scripts.config import DBConf
from scripts.core.engine.oee_calculator import OEETagFinder
from scripts.core.handlers.batch_oee_calc_handler import CalculateBatchOEEHandler
from scripts.core.handlers.common_handler import CommonHandler
from scripts.db.mongo.schema.tag_hierarchy import GetTagsLists, OutputTagsList
from scripts.errors import DataNotFound
from scripts.logging import logger
from scripts.schemas.batch_oee import ChartResponse, ChartDBResponse, ChartRequest, OEEDataInsertRequest
from scripts.utils.common_utils import CommonUtils
from scripts.utils.kairos_db_util import BaseQuery
from scripts.utils.kairos_db_util.df_formation_util import create_kairos_df
from scripts.utils.kairos_db_util.query_kairos import KairosQuery


class OEEAggregator:
    def __init__(self, project_id=None):
        self.common_util = CommonUtils()
        self.base_query = BaseQuery()
        self.oee_tag_finder = OEETagFinder()
        self.common_handler = CommonHandler(project_id=project_id)
        self.oee_handler = CalculateBatchOEEHandler(project_id=project_id)

    def processor(self, data):
        db_response = ChartDBResponse(**data)
        duration = self.common_util.get_duration(meta=data, difference=True, tz=data["tz"])
        cal_type = self.common_util.get_uom_type(uom_type=data["uom"])
        db_response.total_time = self.common_util.get_diff_duration_in_int(input_time=duration, return_type=cal_type)
        db_response.actual_cycle = round(
            db_response.total_units / db_response.total_time, 2
        )
        db_response.ideal_cycle = round(db_response.cycle_time, 2)
        db_response.good_units = round(
            db_response.total_units - db_response.reject_units, 2
        )

        chart_response = ChartResponse(**db_response.dict())
        return chart_response.dict()

    def aggregator(self, request_data: ChartRequest):
        try:
            start_time = int(self.common_util.pendulum_conversion(request_data.queryDate[0], tz=request_data.tz,
                                                                  timestamp=True)) * 1000
            end_time = int(self.common_util.pendulum_conversion(request_data.queryDate[-1], tz=request_data.tz,
                                                                timestamp=True)) * 1000
            hierarchy_tags = self.common_handler.tag_hierarchy_handler.get_tags_list_by_hierarchy(
                GetTagsLists(**request_data.dict()))
            total_units_tag_id = self.oee_tag_finder.get_total_units_tag_id(input_data=hierarchy_tags)
            reject_units_tag_id = self.oee_tag_finder.get_reject_units_tag_id(input_data=hierarchy_tags)
            output_tags_dict = self.common_handler.tag_hierarchy_handler.get_output_tags_for_oee(
                input_data=OutputTagsList(**request_data.dict()))
            if not output_tags_dict or not output_tags_dict.get(request_data.hierarchy):
                return {}
            updated_dict = self.common_handler.validate_hierarchy_tags(output_tags_dict[request_data.hierarchy])
            new_columns_dict = self.common_handler.get_oee_keys_mapping_dict(output_tags_dict[request_data.hierarchy])
            tags_list = list(updated_dict.values())
            group_by_tags_list = deepcopy(tags_list)
            group_by_tags_list.append(DBConf.KAIROS_DEFAULT_FULL_TAG)
            tags_list.extend([total_units_tag_id, reject_units_tag_id])
            if not tags_list:
                return {}
            kairos_util = KairosQuery(url=DBConf.KAIROS_URL)
            data = kairos_util.query(
                self.base_query.form_generic_query(tags_list=tags_list,
                                                   project_id=request_data.project_id,
                                                   start_epoch=start_time, end_epoch=end_time))
            master_df = pd.DataFrame()
            data = [data] if not isinstance(data, list) else data
            for each_data in data:
                master_df = create_kairos_df(
                    master_df=master_df,
                    response_data=each_data,
                    tags_list=tags_list,
                    group_by_tags=group_by_tags_list,
                    tz=request_data.tz
                )
            if master_df.empty:
                raise DataNotFound
            master_df_columns = list(master_df.columns)
            input_data = {"prod_start_time": start_time, "prod_end_time": end_time}
            input_data.update(request_data.dict(exclude_none=True))
            input_schema = OEEDataInsertRequest(**input_data)
            input_schema.total_units, input_schema.reject_units = self.oee_handler.get_data_for_tags(
                input_data=input_schema)


        except Exception as e:
            logger.execption(f'Exception occurred while plotting the dashboard {e.args}')
