if __name__ == '__main__':
    from dotenv import load_dotenv

    load_dotenv()
import os
import time
from datetime import datetime, timedelta

from scripts.constants import CommonConstants, TagCategoryConstants
from scripts.core.engine.oee_calculator import OEEEngine
from scripts.core.handlers.batch_oee_calc_handler import CalculateBatchOEEHandler
from scripts.core.handlers.common_handler import CommonHandler
from scripts.logging import logger
from scripts.schemas.batch_oee import MachineOEERequest, BatchOEEData, OEEDataInsertRequest, OEEDataSaveRequest
from scripts.utils.common_utils import CommonUtils
from scripts.utils.kafka_util import DataPush


class MachineOEECalculator:
    def __init__(self, project_id=None):
        self.common_util = CommonUtils()
        self.batch_oee_handler = CalculateBatchOEEHandler()
        self.common_handler = CommonHandler(project_id=project_id)
        self.oee_engine = OEEEngine()
        self.data_push = DataPush()

    def calculate_machine_oee(self, request_data: MachineOEERequest):

        try:
            hierarchy_dict = self.common_handler.get_valid_oee_monitoring_hierarchy(project_id=request_data.project_id)
            now = datetime.today() - timedelta(days=2)
            oee_start_time = datetime.strptime(request_data.monitor_time, '%H:%M').replace(year=now.year,
                                                                                           month=now.month,
                                                                                           day=now.day).strftime(
                CommonConstants.USER_META_TIME_FORMAT)
            oee_end_time = datetime.now().strftime(CommonConstants.USER_META_TIME_FORMAT)
            for k, v in hierarchy_dict.items():
                site_id = k.split("$")[0]
                downtime = self.common_util.get_downtime_details_by_hierarchy(
                    hierarchy=k, project_id=request_data.project_id)
                input_data = OEEDataInsertRequest(prod_start_time=oee_start_time,
                                                  prod_end_time=oee_end_time, downtime=downtime,
                                                  hierarchy=k, cycle_time=os.environ.get("CYCLE_TIME", default=5),
                                                  tz=request_data.tz,
                                                  project_id=request_data.project_id)
                input_data.total_units, input_data.reject_units = self.batch_oee_handler.get_data_for_tags(
                    input_data=input_data)
                oee_response: BatchOEEData = self.oee_engine.start_batch_oee_calc(
                    request_data=OEEDataSaveRequest(**input_data.dict()))
                data_dict = {
                    v[TagCategoryConstants.OEE_OUTPUT_CATEGORY]: oee_response.oee,
                    v[TagCategoryConstants.OEE_OUTPUT_PERFORMANCE_CATEGORY]: oee_response.performance,
                    v[TagCategoryConstants.OEE_OUTPUT_QUALITY_CATEGORY]: oee_response.quality,
                    v[TagCategoryConstants.OEE_OUTPUT_QUALITY_LOSS_CATEGORY]: oee_response.quality_loss,
                    v[TagCategoryConstants.OEE_OUTPUT_PERFORMANCE_LOSS_CATEGORY]: oee_response.performance_loss,
                    v[TagCategoryConstants.OEE_OUTPUT_AVAILABILITY_CATEGORY]: oee_response.availability,
                    v[TagCategoryConstants.OEE_OUTPUT_AVAILABILITY_LOSS_CATEGORY]: oee_response.availability_loss
                }
                message_dict = {
                    "data": data_dict,
                    "site_id": site_id,
                    "gw_id": "",
                    "pd_id": "",
                    "p_id": request_data.project_id,
                    "timestamp": int(time.time() * 1000),
                    "msg_id": 1,
                    "retain_flag": False
                }
                self.data_push.publish_message(message_dict)

        except Exception as e:
            logger.exception(f"Exception Occurred while calculating oee for the hierarchy {e.args}")
        return


if __name__ == '__main__':
    while True:
        projects_list = os.environ.get("OEE_PROJECTS", default="project_170")
        monitor_start_time = os.environ.get("OEE_START_TIME", default="00:00")
        for project in projects_list.split(","):
            MachineOEECalculator().calculate_machine_oee(
                request_data=MachineOEERequest(project_id=project, monitor_time="00:00",
                                               tz="Asia/Kolkata"))
        time.sleep(10)
