from datetime import datetime
import time
import pendulum
from scripts.constants.db_connections import mongo_client
from scripts.config import PathToServices
from scripts.constants import Secrets, CommonKeys
from scripts.logging import logger
from scripts.utils.auth_util import ILensRequest, AuthenticationError
from scripts.utils.security_utils.apply_encrytion_util import create_token


class CommonUtils:
    def __init__(self, project_id=None):
        self.common_key = CommonKeys()

    @staticmethod
    def check_date_format(date_time_str, time_format):
        try:
            date_time = datetime.strptime(date_time_str, time_format)
        except ValueError:
            raise ValueError("Invalid Date Format")
        if date_time_str != date_time.strftime(time_format):
            raise ValueError("Invalid Date Format")

    @staticmethod
    def get_duration(tz, meta: dict, difference=False):
        try:
            from_time = meta.get("prod_start_time")
            if not from_time:
                return ""
            if not meta.get("prod_end_time"):
                to_time = pendulum.now(tz=tz)
            else:
                to_time = meta.get("prod_end_time")
                to_time = pendulum.parse(to_time, tz=tz)
            from_time = pendulum.parse(from_time, tz=tz)
            diff = to_time - from_time
            if difference:
                return diff
            return f"{int(diff.in_hours())} hours {int(diff.minutes)} minutes"
        except Exception as e:
            logger.exception(f"Exception in getting data: {e}")
            raise

    def get_downtime_details_by_hierarchy(self, hierarchy, project_id, user_id=None):
        connection_obj = ILensRequest(url=PathToServices.downtime_proxy,
                                      project_id=project_id)
        try:
            cookies = {'login-token': self.create_token(user_id=user_id), "user_id": user_id}
            downtime_response = connection_obj.post(path="/downtime_log/get/overall_downtime",
                                                    json={"project_id": project_id,
                                                          "hierarchy": hierarchy})
            response = downtime_response.json()
            return response.get("data", 0)
        except AuthenticationError:
            logger.exception(f"Authentication Error when trying to connect with downtime module")
            return 0
        except Exception as e:
            logger.exception(f'{e.args}')
            return 0

    @staticmethod
    def create_token(host: str = '127.0.0.1', user_id=None, internal_token=Secrets.token, age=2):
        """
        This method is to create a cookie
        """

        try:
            if user_id is None:
                user_id = "user_099"
            new_token = create_token(
                user_id=user_id,
                ip=host,
                token=internal_token,
                age=age
            )
            return new_token
        except Exception as e:
            logger.exception(str(e))
            raise

    def get_user_meta(self, user_id=None, check_flag=False):
        data_for_meta = {}
        if check_flag:
            data_for_meta[self.common_key.KEY_CREATED_BY] = user_id
            data_for_meta[self.common_key.KEY_CREATED_TIME] = int(time.time() * 1000)
            data_for_meta[self.common_key.KEY_UPDATED_AT] = user_id
            data_for_meta[self.common_key.KEY_LAST_UPDATED_TIME] = int(time.time() * 1000)
        else:
            data_for_meta[self.common_key.KEY_UPDATED_AT] = user_id
            data_for_meta[self.common_key.KEY_LAST_UPDATED_TIME] = int(time.time() * 1000)
        return data_for_meta
