from typing import Any, Optional

from pydantic import BaseModel

from scripts.constants import CommonKeys
from scripts.db.mongo.ilens_configuration import database
from scripts.db.mongo.ilens_configuration.collections import collection_constants
from scripts.utils.mongo_util import MongoCollectionBaseClass


class ConstantsSchema(BaseModel):
    """
    This is the Schema for the Mongo DB Collection.
    All datastore and general responses will be following the schema.
    """
    type: Optional[str]
    data: Optional[Any]
    map_json: Optional[Any]
    content_type: Optional[Any]
    content: Optional[Any]


class Constants(MongoCollectionBaseClass):
    def __init__(self, mongo_client):
        super().__init__(mongo_client, database=database,
                         collection=collection_constants)

    @property
    def key_type(self):
        return CommonKeys.KEY_TYPE

    @property
    def key_content_type(self):
        return CommonKeys.KEY_CONTENT_TYPE

    def find_constant_by_dict(self, _type):
        """
        The following function will give one record for a given set of
        search parameters as keyword arguments
        :param _type:
        :return:
        """
        record = self.find_one(query={self.key_type: _type})
        if not record:
            return dict(record)
        return dict(record)

    def find_constant(self, _type, filter_dict=None):
        """
        The following function will give one record for a given set of
        search parameters as keyword arguments
        :param _type:
        :param filter_dict:
        :return:
        """
        query = {self.key_type: _type}
        record = self.find_one(query=query, filter_dict=filter_dict)
        if not record:
            return dict()
        return ConstantsSchema(**record).dict()

    def find_constant_by_query(self, query, filter_dict=None):
        record = self.find_one(query=query, filter_dict=filter_dict)
        if not record:
            return dict()
        return ConstantsSchema(**record).dict()

    def insert_one_constant(self, data):
        """
        The following function will insert one tag in the
        tags collections
        :param self:
        :param data:
        :return:
        """
        return self.insert_one(data)

    def find_constant_by_content(self, content_type):
        """
        The following function will give one record for a given set of
        search parameters as keyword arguments
        """
        query = {self.key_content_type: content_type}
        record = self.find_one(query=query)
        if not record:
            return dict()
        return record

    def find_constant_by_aggregate(self, query):
        constant = self.aggregate(query)
        if not constant:
            return list()
        return list(constant)

    def find_constant_dict(self, _type, filter_dict=None):
        """
        The following function will give one record for a given set of
        search parameters as keyword arguments
        :param _type:
        :param filter_dict:
        :return:
        """
        query = {self.key_type: _type}
        record = self.find_one(query=query, filter_dict=filter_dict)
        if not record:
            return dict()
        return dict(record)
