from datetime import datetime

import pendulum
import pytz
import shortuuid

from scripts.config import PathToServices
from scripts.constants import Secrets, EndpointConstants, UOM, CommonConstants
from scripts.db.redis_connections import project_details_db
from scripts.logging import logger
from scripts.utils.auth_util import ILensRequest, AuthenticationError
from scripts.utils.db_name_util import get_db_name
from scripts.utils.security_utils.apply_encrytion_util import create_token


class CommonUtils:

    @staticmethod
    def check_date_format(date_time_str, time_format):
        try:
            date_time = datetime.strptime(date_time_str, time_format)
        except ValueError:
            raise ValueError("Invalid Date Format")
        if date_time_str != date_time.strftime(time_format):
            raise ValueError("Invalid Date Format")

    @staticmethod
    def get_duration(tz, meta: dict, difference=False):
        try:
            from_time = meta.get("prod_start_time")
            if not from_time:
                return ""
            if not meta.get("prod_end_time"):
                to_time = pendulum.now(tz=tz)
            else:
                to_time = meta.get("prod_end_time")
                to_time = pendulum.parse(to_time, tz=tz)
            from_time = pendulum.parse(from_time, tz=tz)
            diff = to_time - from_time
            if difference:
                return diff
            return f"{int(diff.in_hours())} hours {int(diff.in_hours())} minutes"
        except Exception as e:
            logger.exception(f"Exception in getting data: {e}")
            raise

    def get_downtime_details_by_hierarchy(self, hierarchy, project_id, user_id=None, uom_type="minutes",
                                          filters: dict = None):
        connection_obj = ILensRequest(url=PathToServices.downtime_proxy,
                                      project_id=project_id)
        try:
            cookies = {'login-token': self.create_token(user_id=user_id), "user_id": user_id}
            input_data = {"project_id": project_id, "hierarchy": hierarchy, "display_type": uom_type}
            if filters:
                input_data.update({"filters": filters})
            downtime_response = connection_obj.post(path=EndpointConstants.hierarchy_downtime,
                                                    json=input_data, cookies=cookies)
            response = downtime_response.json()
            return response.get("data", 0)
        except AuthenticationError:
            logger.exception(f"Authentication Error when trying to connect with downtime module")
            return 0
        except Exception as e:
            logger.exception(f'{e.args}')
            return 0

    @staticmethod
    def create_token(host: str = '127.0.0.1', user_id=None, internal_token=Secrets.token, age=2):
        """
        This method is to create a cookie
        """

        try:
            if user_id is None:
                user_id = "user_099"
            new_token = create_token(
                user_id=user_id,
                ip=host,
                token=internal_token,
                age=age
            )
            return new_token
        except Exception as e:
            logger.exception(str(e))
            raise

    @staticmethod
    def ts_from_format_as_timezone(_date, _format, tz):
        date = datetime.strptime(_date, _format) \
                   .astimezone(pytz.timezone(tz)).timestamp() * 1000
        return date

    @staticmethod
    def convert_to_timestamp(x, date_format=None) -> int:
        if not isinstance(x, int):
            if isinstance(x, str):
                x = int(datetime.strptime(x, date_format).timestamp() * 1000)
            else:
                x = int(x.timestamp() * 1000)
        return x

    @staticmethod
    def datetime_from_str(date_str, date_format, tz):
        localized_tz = pytz.timezone(tz)
        datetime_with_tz = datetime.strptime(f"{date_str}", date_format)
        localized_dt = localized_tz.localize(datetime_with_tz)
        return localized_dt

    @staticmethod
    def pendulum_conversion(date_str, tz, output_format=CommonConstants.USER_META_TIME_FORMAT, timestamp=False):
        if timestamp:
            return pendulum.parse(date_str, tz=tz).timestamp()
        return pendulum.parse(date_str, tz=tz).strftime(output_format)

    @staticmethod
    def get_next_id(_=None) -> str:
        return shortuuid.uuid()

    @staticmethod
    def metric_name_by_project(metric, project_id):
        """For DB splitting enabled projects"""
        db_splitting_enabled = get_db_name(project_id=project_id,
                                           database=metric, redis_client=project_details_db)
        return db_splitting_enabled

    @staticmethod
    def get_diff_duration_in_int(input_time: pendulum.Duration, return_type):
        if return_type == "minutes":
            return input_time.in_minutes()
        elif return_type == "seconds":
            return input_time.in_seconds()
        elif return_type == "hours":
            return input_time.in_hours()
        elif return_type == "microseconds":
            return input_time.total_seconds()
        else:
            return input_time.in_minutes()

    @staticmethod
    def get_uom_type(uom_type):
        if uom_type == UOM.minutes:
            divisor = UOM.time_divs.minutes
            cal_type = "minutes"
        elif uom_type == UOM.seconds:
            divisor = UOM.time_divs.seconds
            cal_type = "seconds"
        elif uom_type == UOM.hours:
            divisor = UOM.time_divs.hours
            cal_type = "hours"
        elif uom_type == UOM.millis:
            divisor = UOM.time_divs.millis
            cal_type = "microseconds"
        else:
            divisor = UOM.time_divs.minutes
            cal_type = "minutes"
        return cal_type
