import os
from dataclasses import dataclass, field
from typing import Optional

from fastapi import FastAPI, Depends, APIRouter
from fastapi.middleware.cors import CORSMiddleware

from scripts.db.psql.create_default_tables import create_default_psql_dependencies
from scripts.services import route
from scripts.services.tag_list_services import tag_service_router
from scripts.utils.security_utils.decorators import CookieAuthentication
from scripts.utils.security_utils.signature import SignatureVerificationMiddleware
from scripts.config import Service
auth = CookieAuthentication()


@dataclass
class FastAPIConfig:
    title: str = "OEE SERVICES"
    description: str = "A comprehensive tool for calculating oee"
    docs_url: str = os.environ.get("SW_DOCS_URL")
    redoc_url: str = field(default=None)
    # root_path: str = os.environ.get("MODULE_PROXY", default="/d_oee")
    openapi_url: Optional[str] = os.environ.get("SW_OPENAPI_URL")


app = FastAPI(**FastAPIConfig().__dict__)


if Service.verify_signature in [True, 'True', 'true']:
    app.add_middleware(SignatureVerificationMiddleware)

@app.get("/api/oee_services/health_check")
async def ping():
    return {"status": 200}


SECURE_ACCESS = True if os.environ.get("SECURE_ACCESS") in {'true', 'True', True} else False
if SECURE_ACCESS:
    app.include_router(route, dependencies=[Depends(auth)])
    app.include_router(tag_service_router, dependencies=[Depends(auth)])
else:
    app.include_router(route)
    app.include_router(tag_service_router)

if os.environ.get("ENABLE_CORS") in (True, "true", "True") and os.environ.get(
        "CORS_URLS"
):
    app.add_middleware(
        CORSMiddleware,
        allow_origins=os.environ.get("CORS_URLS").split(","),
        allow_credentials=True,
        allow_methods=["GET", "POST", "DELETE", "PUT"],
        allow_headers=["*"],
    )
    print(os.environ.get("CORS_URLS").split(","))


@app.on_event("startup")
async def startup_event():
    """
    This function is to create default database and tables,
    """
    create_default_psql_dependencies()
