from copy import deepcopy
from dataclasses import dataclass
from typing import List

from scripts.config import DBConf
from scripts.utils.common_utils import CommonUtils


@dataclass
class KairosConstants:
    kairos_std_metric_name: str = 'ilens.live_data.raw'
    kairos_complete_tag_key = "complete_tag"
    kairos_column_names: tuple = ("timestamp", "value",)
    kairos_time_unit: str = "ms"
    kairos_version_api: str = "/api/v1/version"
    kairos_health_check: str = "/api/v1/health/status"
    kairos_query_api: str = "/api/v1/datapoints/query"
    kairos_write_data_api: str = "/api_v1/v1/datapoints"
    kairos_time_keys = ("start_absolute", "end_absolute",)
    kairos_empty_result = {
        "queries": [
            {
                "sample_size": 0, "results":
                [
                    {
                        "name": kairos_std_metric_name, "tags": {}, "values": []
                    }
                ]
            }
        ]
    }
    kairos_query_levels = [{
        "label": "Parameter",
        "value": DBConf.KAIROS_DEFAULT_TAG
    }, {
        "label": "Complete Tag",
        "value": DBConf.KAIROS_DEFAULT_FULL_TAG
    }]
    default_selection = [
        {
            "label": "Live Data",
            "metric": "ilens.live_data.raw"
        },
        {
            "label": "Error",
            "metric": "ilens.live_data.error"
        }
    ]

    # note: this doesn't work perfectly for months (31 days) or years (365 days)
    SECONDS_IN_UNIT = {
        'milliseconds': 0.001,
        'seconds': 1,
        'minutes': 60,
        'hours': 3600,
        'days': 86400,
        'weeks': 604800,
        'months': 2678400,
        'years': 31536000
    }

    # constants used in get_range_needed
    FETCH_BEFORE = 'prepend'
    FETCH_AFTER = 'append'
    FETCH_ALL = 'overwrite'


class BaseQuery:
    def __init__(self):
        self.common_util = CommonUtils()

    metric_query = {
        "tags": {
            f"{DBConf.KAIROS_DEFAULT_FULL_TAG}": [
                "site$dept$equipment$tag"
            ]
        },
        "name": KairosConstants.kairos_std_metric_name,
        "group_by": [
            {
                "name": "tag",
                "tags": [
                    f"{DBConf.KAIROS_DEFAULT_FULL_TAG}"
                ]
            }
        ],
        "aggregators": [
            {
                "name": "last",
                "sampling": {
                    "value": "1",
                    "unit": "hours"
                },
                "align_start_time": True
            }
        ]
    }

    kairos_query = {
        "metrics": [

        ],
        "plugins": [

        ],
        "cache_time": DBConf.KAIROS_CACHE_TIME,
        "start_absolute": 0,
        "end_absolute": 0
    }

    static_kairos_query = {"metrics": [
        {
            "tags": {
                "c2": [
                    "dept_1238"
                ]
            },
            "name": KairosConstants.kairos_std_metric_name,
            "group_by": [
                {
                    "name": "tag",
                    "tags": [
                        "c2"
                    ]
                }
            ]
        }
    ],
        "plugins": [],
        "cache_time": 0
    }

    site_status_query = {
        "metrics": [
            {
                "tags": {
                    "c1": "sitelist"  # parameter
                },
                "name": KairosConstants.kairos_std_metric_name,
                "group_by": [
                    {
                        "name": "tag",
                        "tags": [
                            "c1"
                        ]
                    }
                ],
                "aggregators": "aggregators"  # parameter
            }
        ],
        "plugins": [],
        "cache_time": 0,
        "start_relative": "start_relative"  # parameter
    }

    def form_generic_query(self, tags_list: List, start_epoch: int, end_epoch: int, project_id: str,
                           metric_name=KairosConstants.kairos_std_metric_name, category=DBConf.KAIROS_DEFAULT_FULL_TAG):
        metric_name = self.common_util.metric_name_by_project(metric=metric_name, project_id=project_id)
        group_by_tags = deepcopy(tags_list)
        group_by_tags.append(category)
        query = {
            "metrics": [
                {
                    "tags": {
                        category: tags_list
                    },
                    "name": metric_name,
                    "group_by": [
                        {
                            "name": "tag",
                            "tags": group_by_tags
                        }
                    ]
                }
            ],
            "plugins": [],
            "cache_time": 0,
            "start_absolute": start_epoch,
            "end_absolute": end_epoch
        }
        return query
