from copy import deepcopy

import pandas as pd

from scripts.config import DBConf
from scripts.core.engine.oee_calculator import OEETagFinder, OEECalculator
from scripts.core.handlers.common_handler import CommonHandler
from scripts.db.mongo.schema.tag_hierarchy import GetTagsLists, OutputTagsList
from scripts.errors import DataNotFound
from scripts.logging import logger
from scripts.schemas.batch_oee import ChartResponse, ChartDBResponse, ChartRequest
from scripts.utils.common_utils import CommonUtils
from scripts.utils.kairos_db_util import BaseQuery
from scripts.utils.kairos_db_util.df_formation_util import create_kairos_df
from scripts.utils.kairos_db_util.query_kairos import KairosQuery


class OEEAggregator:
    def __init__(self, project_id=None):
        self.common_util = CommonUtils()
        self.base_query = BaseQuery()
        self.oee_tag_finder = OEETagFinder()
        self.oee_calc = OEECalculator()
        self.common_handler = CommonHandler(project_id=project_id)

    def processor(self, input_data: ChartDBResponse):
        duration = self.common_util.get_duration(meta=input_data.dict(), difference=True, tz=input_data.tz)
        cal_type = self.common_util.get_uom_type(uom_type=input_data.uom)
        input_data.total_time = self.common_util.get_diff_duration_in_int(input_time=duration, return_type=cal_type)
        input_data.actual_cycle = round(
            input_data.total_units / input_data.total_time, 2
        )
        input_data.ideal_cycle = round(input_data.cycle_time, 2)
        input_data.good_units = round(
            input_data.total_units - input_data.reject_units, 2
        )

        chart_response = ChartResponse(**input_data.dict())
        return chart_response.dict()

    def aggregator(self, request_data: ChartRequest):
        try:
            start_time = int(self.common_util.pendulum_conversion(request_data.queryDate[0], tz=request_data.tz,
                                                                  timestamp=True)) * 1000
            end_time = int(self.common_util.pendulum_conversion(request_data.queryDate[-1], tz=request_data.tz,
                                                                timestamp=True)) * 1000
            hierarchy_tags = self.common_handler.tag_hierarchy_handler.get_tags_list_by_hierarchy(
                GetTagsLists(**request_data.dict()))

            output_tags_dict = self.common_handler.tag_hierarchy_handler.get_output_tags_for_oee(
                input_data=OutputTagsList(**request_data.dict()))
            if not output_tags_dict or not output_tags_dict.get(request_data.hierarchy):
                return {}
            cycle_time_tag_id = self.oee_tag_finder.get_cycle_time_tag_id(input_data=hierarchy_tags)
            cycle_time_value = self.common_handler.get_cycle_time_value_from_hierarchy(
                tag_id=cycle_time_tag_id.split("$")[-1])
            if isinstance(cycle_time_value, bool) and not cycle_time_value:
                logger.debug(f"OEE Cycle Design parameter details not found for selected hierarchy")
                raise ValueError("Cycle Design Parameters not found")
            updated_dict = self.common_handler.validate_hierarchy_tags(output_tags_dict[request_data.hierarchy])
            new_columns_dict = self.common_handler.get_oee_keys_mapping_dict(output_tags_dict[request_data.hierarchy])
            tags_list = list(updated_dict.values())
            group_by_tags_list = deepcopy(tags_list)
            group_by_tags_list.append(DBConf.KAIROS_DEFAULT_FULL_TAG)
            if not tags_list:
                return {}
            kairos_util = KairosQuery(url=DBConf.KAIROS_URL)
            data = kairos_util.query(
                self.base_query.form_generic_query(tags_list=tags_list,
                                                   project_id=request_data.project_id,
                                                   start_epoch=start_time, end_epoch=end_time))
            master_df = pd.DataFrame()
            data = [data] if not isinstance(data, list) else data
            for each_data in data:
                master_df = create_kairos_df(
                    master_df=master_df,
                    response_data=each_data,
                    tags_list=tags_list,
                    group_by_tags=group_by_tags_list,
                    tz=request_data.tz,
                    df_diff=False
                )
            if master_df.empty:
                raise DataNotFound
            master_df.rename(columns=new_columns_dict, inplace=True)
            final_records = master_df.to_dict('records')
            oee_json = final_records[-1] if final_records else {}
            oee_json.update({"productive_time": self.oee_calc.calculate_productive_time(cycle_time=cycle_time_value,
                                                                                        units_produced=oee_json[
                                                                                            "total_units"]),
                             "tz": request_data.tz, "cycle_time": cycle_time_value,
                             "prod_start_time": request_data.queryDate[0],
                             "prod_end_time": request_data.queryDate[-1]
                             })

            res = self.processor(input_data=ChartDBResponse(**oee_json))
            return res
        except Exception as e:
            logger.exception(f'Exception occurred while plotting the dashboard {e.args}')
            raise
