"""
This file exposes configurations from config file and environments as Class Objects
"""
import shutil

if __name__ == '__main__':
    from dotenv import load_dotenv

    load_dotenv()
import os.path
import sys
from configparser import ConfigParser, BasicInterpolation


class EnvInterpolation(BasicInterpolation):
    """
    Interpolation which expands environment variables in values.
    """

    def before_get(self, parser, section, option, value, defaults):
        value = super().before_get(parser, section, option, value, defaults)

        if not os.path.expandvars(value).startswith('$'):
            return os.path.expandvars(value)
        else:
            return


try:
    config = ConfigParser(interpolation=EnvInterpolation())
    config.read(f"conf/application.conf")
except Exception as e:
    print(f"Error while loading the config: {e}")
    print("Failed to Load Configuration. Exiting!!!")
    sys.stdout.flush()
    sys.exit()


class Service:
    HOST = config.get("SERVICE", "host")
    PORT = config.getint("SERVICE", "port")
    WORKERS = config.getint("SERVICE", "workers")
    secure_cookie = config.get("SERVICE", "secure_cookie")


class DBConf:
    POSTGRES_URI = config.get('POSTGRES', 'uri')
    if not POSTGRES_URI:
        print("Error, environment variable OEE_POSTGRES_URI not set")
        sys.exit(1)
    MONGO_URI = config.get('MONGO', 'uri')
    if not MONGO_URI:
        print("Error, environment variable MONGO_URI not set")
        sys.exit(1)
    KAIROS_URL = config.get("KAIROS_DB", "uri")
    KAIROS_DEFAULT_FULL_TAG = config.get("KAIROS_DB", "default_full_tag_level")
    KAIROS_DEFAULT_SITE_TAG = config.get("KAIROS_DB", "default_site_tag_level")
    KAIROS_DEFAULT_TAG = config.get("KAIROS_DB", "default_tag_level")
    KAIROS_CACHE_TIME = config.getint("KAIROS_DB", "kairos_cache_time")


class PathToStorage(object):
    BASE_PATH = config.get("DIRECTORY", "base_path")
    if not BASE_PATH:
        print("Error, environment variable BASE_PATH not set")
        sys.exit(1)
    MOUNT_DIR = config.get("DIRECTORY", "mount_dir")
    if not MOUNT_DIR:
        print("Error, environment variable MOUNT_DIR not set")
        sys.exit(1)
    MODULE_PATH = os.path.join(BASE_PATH, MOUNT_DIR)
    FORM_IO_UPLOADS = os.path.join(MODULE_PATH, "form_io_uploads")
    UPLOAD_FILE_PATH = os.path.join(MODULE_PATH, "csv_files")
    LOGS_MODULE_PATH = f"{BASE_PATH}/logs{MOUNT_DIR}/"


class RedisConfig(object):
    REDIS_URI = config.get("REDIS", "uri")
    LOGIN_DB = config["REDIS"]["login_db"]
    PROJECT_TAGS_DB = config.getint("REDIS", "project_tags_db")
    DOWNTIME_DB = config.getint("REDIS", "downtime_db")
    OEE_PRODUCTION_DB = config.getint("REDIS", "oee_production_db")
    LIVE_TAGS_DB = config.getint("REDIS", "live_tags_db")


class KeyPath(object):
    keys_path = config['DIRECTORY']['keys_path']
    if not os.path.isfile(os.path.join(keys_path, "public")) or not os.path.isfile(
            os.path.join(keys_path, "private")):
        if not os.path.exists(keys_path):
            os.makedirs(keys_path)
        shutil.copy(os.path.join("assets", "keys", "public"), os.path.join(keys_path, "public"))
        shutil.copy(os.path.join("assets", "keys", "private"), os.path.join(keys_path, "private"))
    public = os.path.join(keys_path, "public")
    private = os.path.join(keys_path, "private")


class Logging:
    level = config.get("LOGGING", "level", fallback="INFO")
    level = level if level else "INFO"
    tb_flag = config.getboolean("LOGGING", "traceback", fallback=True)
    tb_flag = tb_flag if tb_flag is not None else True


class StoragePaths:
    module_name = config.get('SERVICE', 'module_name')
    if not module_name:
        module_name = "downtime_oee"
    base_path = os.path.join("data", module_name)
    report_path = os.path.join(base_path, "reports")


class MQTTConf:
    host = config["MQTT"]["host"]
    port = int(config["MQTT"]["port"])
    publish_base_topic = config["MQTT"]["publish_base_topic"]


class KafkaConf:
    host = config.get('KAFKA', 'host')
    port = config.get('KAFKA', 'port')
    topic = config.get('KAFKA', 'topic')
    if not any([topic, host, port]):
        print("KAFKA env variables missing, continuing without Kafka/Kairos support")
    enable_sites_partition = config.getboolean("KAFKA", "ENABLE_KAFKA_PARTITION", fallback=True)
    split_key = config["KAFKA"].get('KAFKA_PARTITION_KEY', 'site_id')
    round_robin_enable = config.getboolean("KAFKA", "ROUND_ROBIN_PARTITION", fallback=True)
    redis_db = config.getint("KAFKA", "partition_db")


class PathToServices:
    event_explorer = config["PATH_TO_SERVICES"]["event_explorer"]
    metadata_proxy = config["PATH_TO_SERVICES"]["metadata_proxy"]
    downtime_proxy = config["PATH_TO_SERVICES"]["downtime_proxy"]
