from typing import Optional, Union, List
import time
from pydantic import BaseModel, validator


class GetProducts(BaseModel):
    queryDate: List[int]
    hierarchy: str
    project_id: Optional[str]


class WaterFallChart(BaseModel):
    performance_loss: float
    quality_loss: float
    availability_loss: float
    oee: float

    @validator("*")
    def round_float(cls, v):
        return round(v, 2)


class ChartRequest(BaseModel):
    project_id: str
    queryDate: List[int]
    hierarchy: Optional[str]
    productId: Optional[str]
    aggregation: Optional[bool] = False


class ChartDBResponse(BaseModel):
    total_units: float
    reject_units: int
    oee: int
    availability: float
    downtime: int
    performance: int
    performance_loss: float
    quality: int
    availability_loss: float
    quality_loss: float
    cycle_time: float
    batch_start_time: int
    batch_end_time: int
    good_units: Optional[float]
    actual_cycle: Optional[float]
    ideal_cycle: Optional[float]
    total_time: Optional[float]
    productive_time: int
    downtime: int

    @validator("*")
    def round_float(cls, v):
        return round(v, 2)


class ChartResponse(BaseModel):
    total_units: float
    reject_units: int
    oee: int
    availability: float
    downtime: int
    performance: int
    quality: int
    actual_cycle: float
    ideal_cycle: float
    good_units: float
    availability_loss: float
    quality_loss: float
    performance_loss: float
    total_time: float
    productive_time: int


class DowntimeLogsRequest(BaseModel):
    project_id: str
    start_time: int
    end_time: int
    hierarchy: str


class ProductInfo(BaseModel):
    product_start_time: int
    product_end_time: int
    hierarchy: str
    product_id: str
    setup_time: Optional[Union[float, int]] = 0
    cycle_time: Union[float, int]
    total_units: Union[float, int]
    reject_units: Optional[Union[float, int]] = 0


class BatchOEEDataRequest(BaseModel):
    batch_start_time: Optional[int]
    batch_end_time: Optional[int]
    downtime: Optional[Union[float, int]]
    hierarchy: Optional[str]
    batch_id: Optional[str]
    setup_time: Optional[Union[float, int]]
    cycle_time: Optional[Union[float, int]]
    total_units: Optional[Union[float, int]]
    reject_units: Optional[Union[float, int]]
    uom: Optional[str]


class BatchOEEDataSaveRequest(BaseModel):
    batch_start_time: Optional[int]
    batch_end_time: Optional[int] = int(time.time() * 1000)
    downtime: Optional[Union[float, int]] = 0
    hierarchy: Optional[str]
    batch_id: Optional[str]
    setup_time: Optional[Union[float, int]] = 0
    cycle_time: Union[float, int]
    total_units: Optional[Union[float, int]] = 0
    reject_units: Optional[Union[float, int]] = 0
    uom: Optional[str] = "mins"


class BatchOEEData(BatchOEEDataRequest):
    calculated_on: int
    productive_time: float
    availability: float
    performance: float
    quality: float
    availability_loss: float
    quality_loss: float
    performance_loss: float
    oee: float

    class Config:
        orm_mode = True


class GetOEERequest(BaseModel):
    start_time: int
    end_time: int
    hierarchy: str


class GetOEERequestOneBatch(BaseModel):
    hierarchy: str
    batch_id: str


class GetBatches(GetOEERequest):
    pass


class GetOeeServices(BaseModel):
    hierarchy: str
    project_id: str
    meta: dict
    oee_tag_mapping: str
