import re
from typing import Optional, Any

from scripts.constants import TagHierarchyKeys, DBConstants
from scripts.db.mongo.schema import MongoBaseSchema
from scripts.utils.mongo_util import MongoCollectionBaseClass


class TagHierarchySchema(MongoBaseSchema):
    """
    This is the Schema for the Mongo DB Collection.
    All datastore and general responses will be following the schema.
    """
    project_id: Optional[str]
    id: Optional[str]
    site_id: Optional[str]
    site_name: Optional[str]
    dept_id: Optional[str]
    dept_name: Optional[str]
    line_id: Optional[str]
    line_name: Optional[str]
    equipment_id: Optional[str]
    equipment_name: Optional[str]
    parameter_id: Optional[str]
    lower_limit: Optional[float]
    upper_limit: Optional[float]
    kpi_level_list: Optional[list]
    kpi_parent_list: Optional[list]
    kpi_details: Optional[Any]
    tag_category_id: Optional[list]
    tag_code: Optional[str]
    tag_register: Optional[str]
    product_limits: Optional[dict] = {}
    enable_product: Optional[bool] = False
    starred: Optional[bool] = False
    additional_fields: Optional[dict] = {}
    additional_fields_mapping: Optional[list] = []


class TagHierarchy(MongoCollectionBaseClass):
    def __init__(self, mongo_client, project_id=None):
        super().__init__(mongo_client, database=DBConstants.db_metadata,
                         collection=DBConstants.collection_tag_hierarchy)
        self.project_id = project_id

    @property
    def key_id(self):
        return TagHierarchyKeys.KEY_ID

    @property
    def key_site_id(self):
        return TagHierarchyKeys.KEY_SITE_ID

    @property
    def key_dept_id(self):
        return TagHierarchyKeys.KEY_DEPT_ID

    @property
    def key_line_id(self):
        return TagHierarchyKeys.KEY_LINE_ID

    @property
    def key_equipment_id(self):
        return TagHierarchyKeys.KEY_EQUIPMENT_ID

    @property
    def key_parameter_id(self):
        return TagHierarchyKeys.KEY_PARAMETER_ID

    @property
    def key_dept_name(self):
        return TagHierarchyKeys.KEY_DEPT_NAME

    @property
    def key_site_name(self):
        return TagHierarchyKeys.KEY_SITE_NAME

    @property
    def key_line_name(self):
        return TagHierarchyKeys.KEY_LINE_NAME

    @property
    def key_equipment_name(self):
        return TagHierarchyKeys.KEY_EQUIPMENT_NAME

    @property
    def key_project_id(self):
        return TagHierarchyKeys.KEY_PROJECT_ID

    @property
    def key_tag_code(self):
        return TagHierarchyKeys.KEY_TAG_CODE

    @property
    def key_tag_register(self):
        return TagHierarchyKeys.KEY_TAG_REGISTER

    @property
    def key_tag_category_id(self):
        return TagHierarchyKeys.KEY_TAG_CATEGORY_ID

    @property
    def key_starred(self):
        return TagHierarchyKeys.KEY_STARRED

    @property
    def key_asset_model_id(self):
        return TagHierarchyKeys.KEY_ASSET_MODEL_ID

    @property
    def key_asset_version(self):
        return TagHierarchyKeys.KEY_ASSET_VERSION

    def find_all_tag_hierarchy(self, filter_dict=None,
                               sort=None, skip=0, limit=None, **query):
        """
        The following function will give all tag_hierarchy for the given set of
        search parameters as keyword arguments
        :param filter_dict:
        :param sort:
        :param skip:
        :param limit:
        :param query:
        :return:
        """
        if query is None:
            query = {}
        all_tag_hierarchy = self.find(filter_dict=filter_dict, sort=sort, skip=skip, limit=limit, query=query)
        if not all_tag_hierarchy:
            return list()
        return list(all_tag_hierarchy)

    def find_by_query(self, query, filter_dict=None):
        return list(self.find(query=query, filter_dict=filter_dict))

    def find_one_tag_hierarchy(self, filter_dict=None, **query) -> TagHierarchySchema:
        """
        The following function will give one tag_hierarchy for a given set of
        search parameters as keyword arguments
        :param filter_dict:
        :param query:
        :return:
        """
        tag_hierarchy = self.find_one(filter_dict=filter_dict, query=query)
        if tag_hierarchy:
            return TagHierarchySchema(**tag_hierarchy)
        else:
            return tag_hierarchy

    def insert_one_tag_hierarchy(self, data):
        """
        The following function will insert one tag_hierarchy in the
        tag_hierarchy collections
        :param data:
        :return:
        """
        return self.insert_one(data)

    def insert_many_tag_hierarchy(self, data):
        """
        The following function will insert many tag_hierarchy in the
        tag_hierarchy collection
        :param data:
        :return:
        """
        return self.insert_many(data)

    def update_one_tag_hierarchy(self, data, upsert=False, **query):
        """
        The following function will update one tag_hierarchy in
        tag_hierarchy collection based on the given query
        :param data:
        :param upsert:
        :param query:
        :return:
        """
        return self.update_one(data=data, upsert=upsert, query=query)

    def update_many_tag_hierarchy(self, data, query, upsert=False):
        """
        The following function will update many tag_hierarchy in
        tag_hierarchy collection based on the given query
        :param data:
        :param upsert:
        :param query:
        :return:
        """
        return self.update_many(data=data, upsert=upsert, query=query)

    def delete_many_tag_hierarchy(self, query):
        """
        The following function will delete many tag_hierarchy in
        tag_hierarchy collection based on the given query
        :param query:
        :return:
        """
        return self.delete_many(query=query)

    def delete_one_tag_hierarchy(self, **query):
        """
        The following function will delete one tag_hierarchy in
        tag_hierarchy collection based on the given query
        :param query:
        :return:
        """
        return self.delete_one(query=query)

    def distinct_tag_hierarchy(self, query_key):
        """
        Get a list of distinct values for `key` among all documents
        in the result set of this query.

        :param query_key:
        :return:
        """
        return self.distinct(query_key=query_key)

    def find_tag_hierarchy_by_ids(self, tag_hierarchy_category_id, tag_hierarchy_list):
        query = {"$or": [{self.key_id: tag_hierarchy_category_id}, {self.key_id: {"$in": tag_hierarchy_list}}]}
        tag_hierarchy = self.find(query=query)
        if not tag_hierarchy:
            return list()
        return list(tag_hierarchy)

    def find_tag_code(self, project_id, tag_code, site_id):
        query = {"$and": [{self.key_project_id: project_id}, {self.key_tag_code: tag_code}, {"site_id": site_id}]}
        tag_record = self.find_one(query=query)
        if not tag_record:
            return dict()
        return dict(tag_record)

    def find_tag_hierarchy_by_aggregate(self, query):
        tag_hierarchy = self.aggregate(query)
        if not tag_hierarchy:
            return list()
        return list(tag_hierarchy)

    def get_tag_hierarchy_by_aggregate(self, query: list):
        return list(self.aggregate(pipelines=query))

    def find_by_aggregate(self, query):
        tag_hierarchy = self.aggregate(query)
        if not tag_hierarchy:
            return list()
        return list(tag_hierarchy)

    def find_tag_hierarchy_by_id(self, project_id, tag_id):
        query = {"$and": [{self.key_project_id: project_id}, {self.key_id: tag_id}]}
        tag_hierarchy_record = self.find_one(query=query)
        if not tag_hierarchy_record:
            return dict()
        return dict(tag_hierarchy_record)

    def get_parameter_ids(self, project_id, regex_string):
        query = {self.key_project_id: project_id, 'tag_code': {'$regex': re.escape(regex_string), '$options': 'i'}}
        tag_list = self.distinct('id', query)
        if not tag_list:
            return []
        return tag_list
