from typing import Optional, Dict, List

from scripts.constants.app_constants import DatabaseNames, CollectionNames
from scripts.constants.db_keys import TagInstanceDataKeys
from scripts.db.mongo.schema import MongoBaseSchema
from scripts.utils.mongo_util import MongoCollectionBaseClass


class TagInstanceDataSchema(MongoBaseSchema):
    """
    This is the Schema for the Mongo DB Collection.
    All datastore and general responses will be following the schema.
    """
    hierarchy: str
    project_id: str
    meta: Optional[dict]
    oee_tag_mapping: Optional[str]


class TagInstanceData(MongoCollectionBaseClass):
    def __init__(self, mongo_client, project_id=None):
        super().__init__(mongo_client, database=DatabaseNames.ilens_assistant,
                         collection=CollectionNames.oee_tags)
        self.project_id = project_id

    @property
    def key_project_id(self):
        return TagInstanceDataKeys.KEY_PROJECT_ID

    def find_data_by_project_id(self, project_id: str):
        query = {"project_id": project_id}
        records = self.find_one(query)
        if not records:
            return TagInstanceDataSchema()
        return TagInstanceDataSchema(**records)

    def find_data_project_id(self, project_id):
        query = {"project": project_id}
        record = self.find_one(query)
        if not record:
            return TagInstanceDataSchema()
        return record

    def find_all_project_instance_data(self, **query):
        records = self.find(query)
        if not records:
            return list()
        return records

    def find_all_project_instance_data_by_dict(self, query: dict):
        records = self.find(query)
        if not records:
            return list()
        return records

    def update_hierarchy_data(self, hierarchy, data, upsert=False):
        query = {"hierarchy": hierarchy}
        return self.update_one(data=data, query=query, upsert=upsert)

    def get_tag_data(self, query):
        record = self.find_one(query, filter_dict=None)
        if not record:
            return None
        return record

    def add_tag(self, data):
        self.insert_one(data)
        return TagInstanceDataSchema(**data)

    def insert_one_tag(self, data):
        return self.insert_one(data)

    def add_new_tag(self, data):
        self.insert_one(data.dict())
        return TagInstanceDataSchema(**data.dict()).hierarchy

    def update_tag_data(self, project_id, data, upsert=False):
        query = {"project_id": project_id}
        return self.update_one(data=data, query=query, upsert=upsert)