import time

from sqlalchemy.orm import Session

from scripts.core.engine.chart_creators import ChartMaker
from scripts.core.engine.oee_aggregator import processor, aggregator
from scripts.db.psql.oee_discrete import DiscreteOEE
from scripts.logging import logger
from scripts.schemas.batch_oee import (
    GetOEERequest,
    GetOEERequestOneBatch,
    GetBatches,
    GetProducts,
    ChartRequest,
)
from scripts.schemas.meta import LabelValue


class APIHandler:
    @staticmethod
    def get_oee_all(db: Session, get_oee_request: GetOEERequest):
        table_obj = DiscreteOEE(db=db)
        try:
            data = table_obj.get_oee_data_all(
                prod_start_time=get_oee_request.prod_start_time,
                prod_end_time=get_oee_request.prod_end_time,
                hierarchy=get_oee_request.hierarchy,
            )
            return data
        except Exception:
            raise

    @staticmethod
    def get_oee_batch(db: Session, get_oee_request: GetOEERequestOneBatch):
        table_obj = DiscreteOEE(db=db)
        try:
            data = table_obj.get_oee_data_by_reference_id(
                hierarchy=get_oee_request.hierarchy, reference_id=get_oee_request.reference_id,
                project_id=get_oee_request.project_id
            )
            return data
        except Exception:
            raise

    @staticmethod
    def get_batches(db: Session, request_data: GetBatches):
        table_obj = DiscreteOEE(db=db)
        try:
            if not request_data.prod_end_time:
                request_data.prod_end_time = int(time.time() * 1000)
            data = table_obj.get_batches(
                hierarchy=request_data.hierarchy,
                prod_start_time=request_data.prod_start_time,
                prod_end_time=request_data.prod_end_time,
            )
            return data
        except Exception:
            raise

    @staticmethod
    async def get_products(db: Session, request_data: GetProducts):
        table_obj = DiscreteOEE(db=db)
        try:
            data = table_obj.get_products(
                hierarchy=request_data.hierarchy,
                prod_start_time=request_data.queryDate[0],
                prod_end_time=request_data.queryDate[1],
            )

            return [
                LabelValue(
                    label=each[0], value=each[0], start_time=each[1], end_time=each[2]
                ) if isinstance(each, list) else LabelValue(
                    label=each['batch_id'], value=each['batch_id'],
                    start_time=each['batch_start_time'], end_time=each['batch_end_time']
                )

                for each in data
            ]
        except Exception as e:
            logger.exception(e, exc_info=True)
            raise

    @staticmethod
    async def get_chart_data(db: Session, request_data: ChartRequest):
        table_obj = DiscreteOEE(db=db)
        try:
            if not request_data.hierarchy:
                return dict()
            chart_maker = ChartMaker()
            data = table_obj.get_chart_data(
                hierarchy=request_data.hierarchy,
                prod_start_time=request_data.queryDate[0],
                prod_end_time=request_data.queryDate[1],
                reference_id=request_data.productId,
                aggregation=request_data.aggregation,
            )
            if not request_data.aggregation or len(data) == 1:
                if isinstance(data, list):
                    data = data[0]
                raw_data = processor(data)
                return chart_maker.main_creator(raw_data, overall=False)
            elif len(data) == 0:
                return dict()
            else:
                agg_data = aggregator(data)
                return chart_maker.main_creator(agg_data)

        except Exception as e:
            raise
        finally:
            del table_obj
