import traceback

from fastapi import APIRouter, Depends
from sqlalchemy.orm import Session

from scripts.constants import Endpoints, ResponseCodes
from scripts.core.handlers.api_handler import APIHandler
from scripts.core.handlers.layout_handler import LayoutHandler
from scripts.db.psql.databases import get_db
from scripts.errors import ILensError, DataNotFound
from scripts.logging import logger
from scripts.schemas.batch_oee import BatchesGet, ChartRequest
from scripts.schemas.layout import GetLayoutRequest, SaveLayoutRequest
from scripts.schemas.response_models import DefaultFailureResponse, DefaultResponse
from scripts.utils.security_utils.cookie_decorator import MetaInfoSchema, MetaInfoCookie

api_handler = APIHandler()

ui_service_router = APIRouter(prefix=Endpoints.api_batches, tags=["UI Services"])
get_cookies = MetaInfoCookie()


@ui_service_router.post(Endpoints.api_get)
async def get_all_batches(request_data: BatchesGet, db: Session = Depends(get_db),
                          meta: MetaInfoSchema = Depends(get_cookies)):
    try:
        data = await api_handler.get_batches_info(request_data=request_data, db=db)
        return DefaultResponse(
            data=data,
            status=ResponseCodes.SUCCESS,
            message="Batches fetched successfully",
        )
    except Exception as e:
        tb = traceback.format_exc()
        logger.exception(e)
        logger.exception(tb)
        return DefaultFailureResponse(error=e.args)


@ui_service_router.post(Endpoints.api_chart_data)
async def get_chart_data(request_data: ChartRequest, db: Session = Depends(get_db),
                         meta: MetaInfoSchema = Depends(get_cookies)):
    try:
        data = await api_handler.get_chart_data(request_data=request_data, db=db)
        return DefaultResponse(
            data=data,
            status=ResponseCodes.SUCCESS,
            message="Chart data fetched successfully",
        )
    except DataNotFound:
        return DefaultResponse(message="Data not Found for selected filters", status="info")
    except ILensError as e:
        return DefaultResponse(message=e.args[0], status="info")
    except Exception as e:
        tb = traceback.format_exc()
        logger.exception(e)
        logger.exception(tb)
        return DefaultFailureResponse(error=e.args)


@ui_service_router.post(Endpoints.get_layout)
async def get_layout(request_data: GetLayoutRequest, meta: MetaInfoSchema = Depends(get_cookies)):
    try:
        layout_handler = LayoutHandler(project_id=meta.project_id)
        data = await layout_handler.fetch_layout(layout_request=request_data)
        return DefaultResponse(
            data=data,
            status=ResponseCodes.SUCCESS,
            message="Layout fetched successfully",
        )
    except Exception as e:
        tb = traceback.format_exc()
        logger.exception(e)
        logger.exception(tb)
        return DefaultFailureResponse(error=e.args)


@ui_service_router.post(Endpoints.save_layout)
async def save_layout(request_data: SaveLayoutRequest, meta: MetaInfoSchema = Depends(get_cookies)):
    try:
        layout_handler = LayoutHandler(project_id=meta.project_id)
        data = await layout_handler.save_layout(layout_request=request_data)
        return DefaultResponse(
            data=data, status=ResponseCodes.SUCCESS, message="Layout saved successfully"
        )
    except Exception as e:
        tb = traceback.format_exc()
        logger.exception(e)
        logger.exception(tb)
        return DefaultFailureResponse(error=e.args)
