from sqlalchemy.orm import Session

from scripts.constants import ResponseCodes
from scripts.core.engine.oee_calculator import OEEEngine
from scripts.db.psql.oee_discrete import DiscreteOEE
from scripts.errors import ILensError, ErrorCodes
from scripts.logging import logger
from scripts.schemas.batch_oee import BatchOEEDataRequest, BatchOEEData, GetOEERequestOneBatch, BatchOEEDataSaveRequest
from scripts.schemas.response_models import DefaultResponse

oee_engine = OEEEngine()


class CalculateBatchOEEHandler:

    async def calculate_oee(self, db, product_info: BatchOEEDataRequest):
        table_obj = DiscreteOEE(db=db)
        try:
            record_presence = table_obj.get_oee_data_batch_id(batch_id=product_info.batch_id,
                                                              hierarchy=product_info.hierarchy)

            if not record_presence:
                if not product_info.batch_start_time:
                    raise ILensError(ErrorCodes.ERR005)
                product_info = BatchOEEDataSaveRequest(**product_info.dict(exclude_none=True))
                oee_calculation = await oee_engine.start_batch_oee_calc(product_info=product_info)
                await self.save_oee_data(oee_calculation, db)
                response = DefaultResponse(
                    status=ResponseCodes.SUCCESS,
                    data=oee_calculation,
                    message="OEE saved Successfully",
                )
                return response
            status = await self.update_oee_data(product_info.dict(exclude_none=True), record_presence, db)
            response = DefaultResponse(
                status=ResponseCodes.SUCCESS,
                data=status,
                message="OEE updated Successfully",
            )
            return response
        except Exception as e:
            logger.exception(f"Exception while saving oee record: {e}")
            raise e
        finally:
            del table_obj

    @staticmethod
    async def save_oee_data(oee_data: BatchOEEData, db: Session):
        try:
            oee_table = BatchOEETable(**oee_data.dict())
            table_obj = batch_oee_table.BatchOEETable(db=db)
            table_obj.add_data(oee_table)
            return True
        except Exception as e:
            raise e
        finally:
            del table_obj

    @staticmethod
    async def update_oee_data(product_info: dict, old_record: dict, db: Session):
        try:
            table_obj = batch_oee_table.BatchOEETable(db=db)
            old_record.update(**product_info)
            oee_calculation = await oee_engine.start_batch_oee_calc(
                product_info=BatchOEEDataSaveRequest(**old_record))
            filters = GetOEERequestOneBatch(batch_id=product_info["batch_id"], hierarchy=product_info["hierarchy"])
            table_obj.update_record(filters=filters, update_obj=oee_calculation.dict())
            return True
        except Exception as e:
            raise e
        finally:
            del table_obj
