"""
This file exposes configurations from config file and environments as Class Objects
"""
import shutil

if __name__ == '__main__':
    from dotenv import load_dotenv

    load_dotenv()
import os.path
import sys
from configparser import ConfigParser, BasicInterpolation


class EnvInterpolation(BasicInterpolation):
    """
    Interpolation which expands environment variables in values.
    """

    def before_get(self, parser, section, option, value, defaults):
        value = super().before_get(parser, section, option, value, defaults)

        if not os.path.expandvars(value).startswith('$'):
            return os.path.expandvars(value)
        else:
            return


try:
    config = ConfigParser(interpolation=EnvInterpolation())
    config.read(f"conf/application.conf")
except Exception as e:
    print(f"Error while loading the config: {e}")
    print("Failed to Load Configuration. Exiting!!!")
    sys.stdout.flush()
    sys.exit()


class Service:
    HOST = config.get("service", "host")
    PORT = config.getint("service", "port")
    WORKERS = config.getint("service", "workers")


class DBConf:
    POSTGRES_URI = config.get('postgres', 'uri')
    if not POSTGRES_URI:
        print("Error, environment variable OEE_POSTGRES_URI not set")
        sys.exit(1)
    DOWNTIME_URI = config.get('postgres', 'downtime_uri')
    if not DOWNTIME_URI:
        print("Error, environment variable DOWNTIME_URI not set")
        sys.exit(1)
    MONGO_URI = config.get('mongo', 'uri')
    if not MONGO_URI:
        print("Error, environment variable MONGO_URI not set")
        sys.exit(1)


class PathToStorage(object):
    BASE_PATH = config.get("DIRECTORY", "base_path")
    if not BASE_PATH:
        print("Error, environment variable BASE_PATH not set")
        sys.exit(1)
    MOUNT_DIR = config.get("DIRECTORY", "mount_dir")
    if not MOUNT_DIR:
        print("Error, environment variable MOUNT_DIR not set")
        sys.exit(1)
    MODULE_PATH = os.path.join(BASE_PATH, MOUNT_DIR)
    FORM_IO_UPLOADS = os.path.join(MODULE_PATH, "form_io_uploads")
    UPLOAD_FILE_PATH = os.path.join(MODULE_PATH, "csv_files")
    LOGS_MODULE_PATH = f"{BASE_PATH}/logs{MOUNT_DIR}/"


class RedisConfig(object):
    uri = config.get("REDIS", "uri")
    login_db = config["REDIS"]["login_db"]
    project_tags_db = config.getint("REDIS", "project_tags_db")
    downtime_db = config.getint("REDIS", "downtime_db")


class KeyPath(object):
    keys_path = config['DIRECTORY']['keys_path']
    if not os.path.isfile(os.path.join(keys_path, "public")) or not os.path.isfile(
            os.path.join(keys_path, "private")):
        if not os.path.exists(keys_path):
            os.makedirs(keys_path)
        shutil.copy(os.path.join("assets", "keys", "public"), os.path.join(keys_path, "public"))
        shutil.copy(os.path.join("assets", "keys", "private"), os.path.join(keys_path, "private"))
    public = os.path.join(keys_path, "public")
    private = os.path.join(keys_path, "private")


class Logging:
    level = config.get("LOGGING", "level", fallback="INFO")
    level = level if level else "INFO"
    tb_flag = config.getboolean("LOGGING", "traceback", fallback=True)
    tb_flag = tb_flag if tb_flag is not None else True


class StoragePaths:
    module_name = config.get('SERVICE', 'module_name')
    if not module_name:
        module_name = "downtime_oee"
    base_path = os.path.join("data", module_name)
    report_path = os.path.join(base_path, "reports")


class MQTTConf:
    host = config["MQTT"]["host"]
    port = int(config["MQTT"]["port"])
    publish_base_topic = config["MQTT"]["publish_base_topic"]
