import os
import time
from datetime import datetime, timezone

from scripts.constants import Constants, Conf
from scripts.logging.logging import logger as LOG
from scripts.utils import Messages
from scripts.utils.encryption_utility import AESCipher


class CommonUtils(Constants):
    def __init__(self):
        pass

    @staticmethod
    def get_utc_datetime_now():
        return datetime.utcnow()

    def ui_datetime_format(self):
        return self.__ui_datetime_format__

    def utc_datetime_format(self):
        return self.__utc_datetime_format__

    def no_special_chars_datetime_format(self):
        return self.__no_special_chars_datetime_format__

    def get_datetime_str(self, dt=None, dt_format='utc_datetime_format'):
        """
        Returns the datetime string.
        :param dt: The datetime value to be converted as str. If not passes, Current UTC time will be returned.
        :param dt_format: The format to which the date string should be converted. 'utc_datetime_format' is the default.
                Supported values are:- 'ui_datetime_format', 'utc_datetime_format', 'no_special_chars_datetime_format'
        :return: Datetime string
        """
        if dt is None:
            dt = self.get_utc_datetime_now()
        return dt.strftime(getattr(self, dt_format)())

    def get_datetime_dt(self, dt_str, dt_format='utc_datetime_format'):
        return datetime.strptime(date_string=dt_str, format=getattr(self, dt_format)())

    @staticmethod
    def get_epoch_now(_round=True):
        if _round:
            return int(time.time())
        else:
            return time.time()

    @staticmethod
    def system_timezone():
        return datetime.now(timezone.utc).astimezone().tzname()

    @staticmethod
    def local_time_offset(t=None):
        if t is None:
            t = time.time()

        if time.localtime(t).tm_isdst and time.daylight:
            return -time.altzone
        else:
            return -time.timezone

    def tz_offset_hm(self):
        seconds = self.local_time_offset()
        hour = seconds // 3600
        seconds %= 3600
        minutes = seconds // 60
        seconds %= 60
        return "%d:%02d:%02d" % (hour, minutes, seconds)

    @staticmethod
    def delete_file(file_name):
        try:
            os.remove(file_name)
            return True
        except OSError as e:
            print(e)
            return False

    @staticmethod
    def write_to_file(file_name, content, overwrite=True):
        try:
            if overwrite:
                write_mode = 'w+'
            else:
                write_mode = 'a+'
            f = open(file_name, write_mode)
            f.write(content)
            f.close()
            return True
        except Exception as e:
            raise Exception(f'Failed to write contents to file: {e}')

    @staticmethod
    def read_from_file(file_name):
        try:
            f = open(file_name, 'r+')
            content = f.read()
            f.close()
            return content
        except Exception as e:
            raise Exception(f'Failed to read contents from file: {e}')

    @staticmethod
    def db_host():
        return Conf.DB_HOST

    @staticmethod
    def db_port():
        return int(Conf.DB_PORT)

    @staticmethod
    def db_type():
        return Conf.DB_TYPE

    @staticmethod
    def db_name():
        return Conf.DB_NAME

    @staticmethod
    def db_username():
        if 'ENC(' in Conf.DB_USERNAME:
            aes_obj = AESCipher(key=Conf.username_decryption_key)
            enc_username = Conf.DB_USERNAME.lstrip('ENC(').rstrip(')')
            username = aes_obj.decrypt(enc=enc_username)
            return username
        else:
            return Conf.DB_USERNAME

    @staticmethod
    def db_password():
        if 'ENC(' in Conf.DB_PASSWORD:
            aes_obj = AESCipher(key=Conf.password_decryption_key)
            enc_password = Conf.DB_PASSWORD.lstrip('ENC(').rstrip(')')
            password = aes_obj.decrypt(enc=enc_password)
            return password
        else:
            return Conf.DB_PASSWORD

    def sqlite_db_endpoint(self):
        LOG.debug("DB type SQLite configured")
        if self.db_host() is None:
            LOG.debug(f"DB path/host not configured. Falling back to default path '{Conf.SQLITE_DEFAULT_DB_PATH}'")
            return f'{self.db_type()}://{Conf.SQLITE_DEFAULT_DB_PATH}'
        else:
            LOG.debug(f"Using DB from user given path '{self.db_host()}'")
            return f'{self.db_type()}://{self.db_host()}'

    def mysql_db_endpoint(self):
        if self.db_username() is None or self.db_password() is None:
            LOG.debug(Messages.MESSAGE01)
            return f'{self.db_type()}://{self.db_host()}/{self.db_name()}'
        else:
            LOG.debug(Messages.MESSAGE01)
            return f'{self.db_type()}://{self.db_username()}:{self.db_password()}@{self.db_host()}/{self.db_name()}'

    def postgresql_db_endpoint(self):
        if self.db_username() is None or self.db_password() is None:
            LOG.debug(Messages.MESSAGE01)
            return f'{self.db_type()}://{self.db_host()}/{self.db_name()}'
        else:
            LOG.debug(Messages.MESSAGE01)
            return f'{self.db_type()}://{self.db_username()}:{self.db_password()}@{self.db_host()}/{self.db_name()}'

    def oracle_db_endpoint(self):
        if self.db_username() is None or self.db_password() is None:
            LOG.debug(Messages.MESSAGE01)
            return f'{self.db_type()}://{self.db_host()}/{self.db_name()}'
        else:
            LOG.debug(Messages.MESSAGE01)
            return f'{self.db_type()}://{self.db_username()}:' \
                   f'{self.db_password()}@{self.db_host()}/{self.db_name()}'

    def mssql_db_endpoint(self):
        if self.db_username() is None or self.db_password() is None:
            LOG.debug(Messages.MESSAGE01)
            return f'{self.db_type()}+pymssql://{self.db_host()}/{self.db_name()}'
        else:
            LOG.debug(Messages.MESSAGE01)
            return f'{self.db_type()}+' \
                   f'pymssql://{self.db_username()}:{self.db_password()}@{self.db_host()}/{self.db_name()}'

    def db_endpoint(self):
        return getattr(self, f'{self.db_type()}_db_endpoint')()

    @staticmethod
    def sqlalchemy_echo():
        if Conf.LOG_LEVEL.lower() == 'trace':
            return True
        else:
            return False

    @staticmethod
    def enable_traceback():
        return Conf.LOG_ENABLE_TRACEBACK

    @staticmethod
    def db_uri():
        return Conf.URI

    @staticmethod
    def db_mongo_name():
        return Conf.METADATA_DB

    @staticmethod
    def db_auth_mechanism():
        return Conf.AUTH_MECHANISM

    @staticmethod
    def db_auth_source():
        return Conf.AUTH_SOURCE

    @staticmethod
    def db_encryption_constants_file_path():
        return Conf.MONGO_ENCRYPTION_FILE_PATH

    @staticmethod
    def server_host():
        return Conf.HOST

    @staticmethod
    def server_port():
        return Conf.PORT

    @staticmethod
    def cross_origin():
        return Conf.ALLOW_CROSS_ORIGIN

    @staticmethod
    def server_threads():
        return Conf.THREADS

    @staticmethod
    def server_workers():
        return Conf.WORKERS

    @staticmethod
    def key_encrypt_keys():
        return Constants.key_encrypt_keys

    @staticmethod
    def key_exclude_encryption():
        return Constants.key_exclude_encryption

    @staticmethod
    def product_encrypted():
        return Constants.product_encrypted

    def max_docs_per_batch(self):
        return Constants.max_docs_per_batch
