"""
This file exposes configurations from config file and environments as Class Objects
"""
import shutil

if __name__ == '__main__':
    from dotenv import load_dotenv

    load_dotenv()
import os.path
import sys
from configparser import ConfigParser, BasicInterpolation


class EnvInterpolation(BasicInterpolation):
    """
    Interpolation which expands environment variables in values.
    """

    def before_get(self, parser, section, option, value, defaults):
        value = super().before_get(parser, section, option, value, defaults)

        if not os.path.expandvars(value).startswith('$'):
            return os.path.expandvars(value)
        else:
            return


try:
    config = ConfigParser(interpolation=EnvInterpolation())
    config.read(f"conf/application.conf")
except Exception as e:
    print(f"Error while loading the config: {e}")
    print("Failed to Load Configuration. Exiting!!!")
    sys.stdout.flush()
    sys.exit()


class Service:
    HOST = config.get("SERVICE", "host")
    PORT = config.getint("SERVICE", "port")
    LOG_ENABLE_TRACEBACK: bool = config.getboolean('SERVICE', 'enable_traceback', fallback=False)



class DBConf:
    MONGO_URI = config.get('MONGO_DB', 'uri')
    if not MONGO_URI:
        print("Error, environment variable MONGO_URI not set")
        sys.exit(1)



class DatabaseConstants:
    ilens_clients_db = config.get("DATABASES", "ilens_clients_db")
    if not bool(ilens_clients_db):
        ilens_clients_db = "ilens_clients"
    ilens_clients_collection = config.get("DATABASES", "ilens_clients_collection")
    if not bool(ilens_clients_collection):
        ilens_clients_collection = "ilens_clients"




class RedisConfig(object):
    host = config["REDIS"]["host"]
    port = int(config["REDIS"]["port"])
    login_db = config["REDIS"]["login_db"]


