import shutil
import time

if __name__ == "__main__":
    from dotenv import load_dotenv

    load_dotenv()
import argparse
import logging
import os
import sys

import gitlab
import jinja2
import ruamel.yaml

from scripts.core import ILensVersionHandler
from scripts.core.git_handler import GitHandler
from scripts.db.psql.databases import get_db_for_func
from scripts.schemas import GetRequest

default_link = "https://gitlab-pm.knowledgelens.com/"
git_user_name = os.environ.get("GIT_USERNAME", default="harshavardhan.c")
git_access_token = os.environ.get("GIT_TOKEN", default="FEMA6PnP63fJCs6DrtZJ")
config_variables = os.environ.get("CONFIG_MAP_VARIABLES", default="").split(",")
helm_repo = os.environ.get("HELM_REPO", default="")
global_configmap = os.environ.get("GLOBAL_VARIABLES_FILE", default="ilens-global-configmap.yml")

HELM_PATH = "/ilens-core/ilens-modules"
HELM_STORE_PATH = "./helm-charts"

git_handler_obj = GitHandler(user_name=git_user_name, access_token=git_access_token)


def render_helm_chart(data_dict, helm_template_file, template_path, outfile_path):
    try:
        environment = jinja2.Environment(
            loader=jinja2.FileSystemLoader(searchpath=template_path),
            trim_blocks=True,
            variable_start_string='<{', variable_end_string='}>')
        _render = environment.get_template(helm_template_file).render(**data_dict)
        with open(outfile_path, "w") as fp:
            fp.write(_render)

    except Exception as e:
        logging.exception(f"Exception occurred while rendering the helm file  - {e.args}")


def convert_yaml_to_json(yaml_file_path):
    try:
        if not os.path.exists(yaml_file_path):
            return {}
        _yaml = ruamel.yaml.YAML(typ='safe')
        _yaml.preserve_quotes = True
        with open(yaml_file_path) as fpi:
            yaml_dict = _yaml.load(fpi)
        return yaml_dict
    except Exception as e:
        logging.exception(f"Exception Occurred while reading the yaml file {e.args}")
        return {}


def convert_json_to_yaml(json_data: dict, output_file_path):
    try:
        yaml = ruamel.yaml.YAML()
        yaml.preserve_quotes = True
        with open(output_file_path, 'w') as yaml_file:
            yaml.dump(json_data, yaml_file)
    except Exception as e:
        logging.exception(f"Exception Occurred while reading the yaml file {e.args}")
        return False


def push_helm_deployments(repo_link: str, private_token: str, branch: str, final_helm_path, base_path: str):
    try:
        base_url = os.environ.get("GIT_BASE_URL", default=default_link)
        repo_link_split = repo_link.split(base_url)
        if not repo_link_split:
            return False
        gl = gitlab.Gitlab(url=base_url, private_token=private_token)
        search_str = repo_link_split[-1].replace(".git", "")
        pl = gl.projects.list(search=search_str.split("/")[-1])
        if not pl:
            return False
        pl = pl[0]
        commit_actions = []
        files_list = os.listdir(final_helm_path)
        branches = pl.branches.list()
        
        if not files_list:
            logging.debug('Files not found for pushing to git.')
        for file in files_list:
            _action = {
                'action': 'create',
                'file_path': os.path.join(HELM_PATH, file),
                'content': open(f'{final_helm_path}/{file}').read()
            }
            commit_actions.append(_action)
        commit_data = {'branch': branch, 'commit_message': f"{branch} helm creation"} | {'actions': commit_actions}

        pl.commits.create(commit_data)
    except Exception as e:
        logging.exception(f'Exception while pushing helm deployments: {e.args}')


ap = argparse.ArgumentParser()
db_handler = ILensVersionHandler()
if __name__ == '__main__':
    ap.add_argument(
        "--ilens_version",
        "-iv",
        required=False,
        default=None,
        help="ILens Version Tag",
    )
    ap.add_argument(
        "--release_version",
        "-rv",
        required=False,
        default=None,
        help="ILens Release Tag",
    )
    ap.add_argument(
        "--client_name",
        "-cn",
        required=False,
        default=None,
        help="Client Name Tag"
    )
    ap.add_argument(
        "--branch_name",
        "-b",
        required=False,
        default=None,
        help="Branch Name"
    )
    ap.add_argument(
        "--git_repos",
        "-gr",
        required=False,
        default=None,
        help="Git repos to be added in helm",
        nargs="+"
    )
    ap.add_argument(
        "--module_names",
        "-mn",
        required=False,
        default=None,
        help="Module names to be added in helm",
        nargs="+"
    )
    HELM_TEMP_PATH = f"{int(time.time())}_helm_tmp_path"
    GENERAL_TEMP_PATH = f"{int(time.time())}_tmp"
    OUTPUT_PATH = f"{int(time.time())}_helm-charts"
    try:
        arguments = vars(ap.parse_args())
        _ilens_version = arguments["ilens_version"]
        _release_version = arguments["release_version"]
        _client_name = arguments['client_name']
        _git_repos = arguments["git_repos"]
        _module_names = arguments["module_names"]
        _branch_name = arguments['branch_name'] or "master"
        if not _ilens_version or not _release_version or not _client_name or not (
                _git_repos or _module_names) or not global_configmap:
            print(
                "global_configmap, git_repos, module_names, client_name, ilens_version and release_version details not found!!!!!")
            sys.exit()
        _branch = f"{_client_name}_{_ilens_version}.{_release_version}"
        if not os.path.exists(HELM_TEMP_PATH):
            os.makedirs(HELM_TEMP_PATH)
        if not os.path.exists(OUTPUT_PATH):
            os.makedirs(OUTPUT_PATH)
        helm_path = os.path.join(HELM_TEMP_PATH, "helm-charts")
        if not git_handler_obj.clone_repository(repo_link=helm_repo, module_output_path=helm_path,
                                                clone_branch=_client_name):
            logging.error(f"Cannot clone helm repo with branch: {_branch}")
            sys.exit()
        base_helm_directory_path = os.path.join(HELM_TEMP_PATH, "helm-charts", "ilens-core", "ilens-modules")
        # global_config_data = convert_yaml_to_json(os.path.join(base_helm_directory_path, global_configmap))
        global_config_data = convert_yaml_to_json("ilens-global-configmap.yml")

        variables_file = "variables.yml"

        for _module in _module_names:
            module_path = os.path.join(GENERAL_TEMP_PATH)
            module_path = os.path.join(module_path, _module)
            if not os.path.exists(module_path):
                os.makedirs(module_path)
            variables_file_path = os.path.join(module_path, variables_file)
            git_info = git_handler_obj.get_git_url_by_module_name(module_name=_module)
            if not git_info:
                logging.debug("Failed to fetch module info!! Skipping Helm File Preparation")
                continue
            if not git_handler_obj.clone_repository_with_defined_file(repo_link=git_info, clone_branch=_branch_name,
                                                                      file_output_path=variables_file_path,
                                                                      clone_file_path=variables_file):
                logging.debug("Failed to clone module!! Skipping Helm File Preparation")
                continue
            existing_yml_path = os.path.join(base_helm_directory_path, f'{_module}.yml')
            if not os.path.exists(existing_yml_path):
                logging.debug(f"{existing_yml_path} not found!! Skipping Helm File Preparation")
                continue
            existing_data = convert_yaml_to_json(existing_yml_path)
            _module_data = convert_yaml_to_json(variables_file_path)
            global_config_vars = global_config_data.get('data', {})
            module_env_variables = _module_data.get('deployment', {}).get('environmentVar', [])
            module_env_variables = {_v['name']: _v for _v in module_env_variables}
            existing_env_variables = {_v['name']: _v for _v in
                                      existing_data.get('deployment', {}).get('environmentVar', [])}
            existing_data['deployment']['environmentVar'] = []
            for k, v in module_env_variables.items():
                if k.lower() in {'port', 'service_port', 'module_port'} and v.get("value"):
                    global_config_vars["SERVICE_PORT"] = v['value']
                    continue
                if "valueFrom" in existing_env_variables.get(k, []):
                    existing_data['deployment']['environmentVar'].append(existing_env_variables[k])
                else:
                    existing_data['deployment']['environmentVar'].append(v)
            template_path = os.path.join(GENERAL_TEMP_PATH, "templates")
            if not os.path.exists(template_path):
                os.makedirs(template_path)
            template_file = os.path.join(template_path, f'{_module}.yml')
            convert_json_to_yaml(json_data=existing_data, output_file_path=template_file)
            session_obj = get_db_for_func()
            module_info = db_handler.get_module_versions(
                input_data=GetRequest(module_name=_module, client='iLens', ilens_version=_ilens_version,
                                      release_version=_release_version), db=session_obj)
            session_obj.close()
            existing_data['deployment']['imageName'] = module_info.get("image_tag",
                                                                       existing_data['deployment']['imageName'])
            helm_out_path = os.path.join(OUTPUT_PATH, f'{_module}.yml')
            render_helm_chart(data_dict=global_config_vars, helm_template_file=f'{_module}.yml',
                              outfile_path=helm_out_path, template_path=template_path)
        push_helm_deployments(helm_repo, git_access_token, _branch, final_helm_path=OUTPUT_PATH, base_path=helm_path)

    except Exception as e:
        logging.exception(f"Exception Occurred while processing the Helm-Script Preparation {e.args}")
    finally:
        # shutil.rmtree(HELM_STORE_PATH)
        shutil.rmtree(HELM_TEMP_PATH)
        shutil.rmtree(GENERAL_TEMP_PATH)
