import os

import git
import gitlab

from scripts.logging import logging

default_link = "https://gitlab-pm.knowledgelens.com/"
HELM_PATH = "/ilens-core/ilens-modules"
HELM_STORE_PATH = "./helm-charts"


class GitHandler:

    def __init__(self, access_token, user_name):
        self.access_token = access_token
        self.user_name = user_name

    def clone_repository(self, repo_link, module_output_path, clone_branch):

        try:
            if repo_link.split("https://")[-1].startswith("gitlab-pm"):
                repo_link = repo_link.replace("https://", f"https://{self.user_name}:{self.access_token}@")
            repo = git.Repo.clone_from(repo_link, module_output_path,
                                       branch=clone_branch)
            return True
        except Exception as e:
            logging.exception(f"Exception occurred while cloning the git repo - {repo_link} - {e.args}")
            return False

    def clone_repository_with_defined_file(self, repo_link: str, file_output_path, clone_branch, clone_file_path):
        try:
            base_url = os.environ.get("GIT_BASE_URL", default=default_link)
            repo_link_split = repo_link.split(base_url)
            if not repo_link_split:
                return False
            gl = gitlab.Gitlab(url=base_url, private_token=self.access_token)
            search_str = repo_link_split[-1].replace(".git", "")
            pl = gl.projects.list(search=search_str)
            if not pl:
                return False
            pl = pl[0]
            with open(file_output_path, 'wb') as f:
                pl.files.raw(file_path=clone_file_path, ref=clone_branch, streamed=True, action=f.write)
            return True
        except gitlab.exceptions.GitlabGetError as e:
            logging.debug(f"Variables.yml not found for selected the module {repo_link}")
            return False
        except Exception as e:
            logging.exception(f"Exception occurred while cloning the git repo - {repo_link} - {e.args}")
            return False

    def get_git_url_by_module_name(self, module_name):
        try:
            base_url = os.environ.get("GIT_BASE_URL", default=default_link)
            gl = gitlab.Gitlab(url=base_url, private_token=self.access_token)
            pl = gl.groups.list(search="iLens-2.0")[0].projects.list(search=module_name,
                                                                     include_subgroups=True)
            return pl[-1].web_url if pl else ''
        except Exception as e:
            logging.exception(f"Exception occurred while fetching repo details - {e.args}")
            return False

    def pull_global_config(self, repo_link: str, branch: str, file_name: str):
        base_url = os.environ.get("GIT_BASE_URL", default=default_link)
        repo_link_split = repo_link.split(base_url)
        if not repo_link_split:
            return False
        gl = gitlab.Gitlab(url=base_url, private_token=self.access_token)
        search_str = repo_link_split[-1].replace(".git", "")
        if pl := gl.projects.list(search=search_str):
            pl = pl[0]
        else:
            return False
        with open(file_name, 'wb') as f:
            pl.files.raw(file_path=f'{HELM_PATH[1:]}/{file_name}', ref=branch, streamed=True, action=f.write)
        return True

    def remove_all_files_from_repo(self, repo_link: str, branch: str, exclude_file: list):
        base_url = os.environ.get("GIT_BASE_URL", default=default_link)
        repo_link_split = repo_link.split(base_url)
        if not repo_link_split:
            return False
        gl = gitlab.Gitlab(url=base_url, private_token=self.access_token)
        search_str = repo_link_split[-1].replace(".git", "")
        if pl := gl.projects.list(search=search_str):
            pl = pl[0]
        else:
            return False
        items = pl.repository_tree(path=HELM_PATH, ref=branch)
        print(items)

    def create_merge_request(self, repo_link: str, source_branch: str, destination_branch: str):
        base_url = os.environ.get("GIT_BASE_URL", default=default_link)
        repo_link_split = repo_link.split(base_url)
        if not repo_link_split:
            return False
        gl = gitlab.Gitlab(url=base_url, private_token=self.access_token)
        search_str = repo_link_split[-1].replace(".git", "")
        if pl := gl.projects.list(search=search_str):
            pl = pl[0]
        else:
            return False
        pl.mergerequests.create({
            'source_branch': source_branch,
            'target_branch': destination_branch,
            'title': f'{source_branch} merge'
        })

    @staticmethod
    def push_deployments_to_git(repo_link: str, private_token: str, branch: str, folder_path, base_path: str,
                                helm_deployment=True):
        try:
            base_url = os.environ.get("GIT_BASE_URL", default=default_link)
            repo_link_split = repo_link.split(base_url)
            if not repo_link_split:
                return False
            gl = gitlab.Gitlab(url=base_url, private_token=private_token)
            search_str = repo_link_split[-1].replace(".git", "")
            pl = gl.projects.list(search=search_str)
            if not pl:
                return False
            pl = pl[0]
            commit_actions = []
            files_list = os.listdir(folder_path)
            branches = pl.branches.list(get_all=True)
            branches_names = [x.name for x in branches]

            if branch not in branches_names:
                pl.branches.create({'branch': branch,
                                    'ref': 'master' if branch.split("_")[0] not in branches_names else
                                    branch.split("_")[
                                        0]})
            if not files_list:
                logging.debug('Files not found for pushing to git.')
            for file in files_list:
                file_path = f'{file}'
                if helm_deployment:
                    file_path = f"{HELM_PATH}/{file}"
                _action = {
                    'action': 'update',
                    'file_path': file_path,
                    'content': open(f'{folder_path}/{file}').read()
                }
                commit_actions.append(_action)
            commit_data = {'branch': branch, 'commit_message': f"{branch} helm creation"} | {'actions': commit_actions}

            pl.commits.create(commit_data)
        except Exception as e:
            logging.exception(f'Exception while pushing deployments to git: {e.args}')
