import os

from scripts.config import EnvironmentVariables
from scripts.core import ILensVersionHandler
from scripts.core.git_handler import GitHandler
from scripts.db.psql.databases import get_db_for_func
from scripts.logging import logging
from scripts.schemas import DockerComposeSchema, GetRequest
from scripts.utils.common_utils import CommonUtils


class DockerHandler:

    def __init__(self, git_user_name, git_access_token):
        self.git_handler_obj = GitHandler(user_name=git_user_name, access_token=git_access_token)
        self.common_util = CommonUtils()
        self.db_handler = ILensVersionHandler()

    def process_module_request(self, compose_data: dict, variables_file, general_path, image_mapping_dict: dict,
                               **kwargs):
        try:
            ilens_version = kwargs.get("ilens_version")
            client_version = kwargs.get("client", "iLens")
            branch_name = kwargs.get("branch_name")
            release_version = kwargs.get("release_version")
            for _module, module_dict in compose_data['services'].items():
                updated_module_name = self.get_defined_module_name(module_name=_module,
                                                                   module_mapping_dict=image_mapping_dict)
                module_path = os.path.join(general_path)
                module_path = os.path.join(module_path, _module)
                if not os.path.exists(module_path):
                    os.makedirs(module_path)
                variables_file_path = os.path.join(module_path, variables_file)
                git_info = self.git_handler_obj.get_git_url_by_module_name(module_name=updated_module_name)
                if not git_info:
                    logging.debug("Failed to fetch module info!! Skipping Helm File Preparation")
                    continue
                if not self.git_handler_obj.clone_repository_with_defined_file(repo_link=git_info,
                                                                               clone_branch=branch_name,
                                                                               file_output_path=variables_file_path,
                                                                               clone_file_path=variables_file):
                    logging.debug("Failed to clone module!! Skipping Docker File Preparation")
                    continue
                _module_data = self.common_util.convert_yaml_to_define_obj(variables_file_path)
                if updated_module_name in image_mapping_dict:
                    image_url = image_mapping_dict.get(updated_module_name, '')
                else:
                    image_url = self.get_image_tag_from_db(module=_module, ilens_version=ilens_version,
                                                           release_version=release_version, client=client_version)
                module_env_variables = _module_data.get('deployment', {}).get('environmentVar', [])
                module_env_variables = {_v['name']: _v.get('value') for _v in module_env_variables}
                image_mapping_dict[_module] = image_url
                existing_env_variables = module_dict.get('environment', {})
                diff_keys = list(
                    set(existing_env_variables.keys()).symmetric_difference(set(module_env_variables.keys())))
                for _each in diff_keys:
                    value = module_env_variables.get(_each)
                    if _each.lower() in {'port', 'service_port', 'module_port'} or not value:
                        continue
                    if value.startswith('{{') and value.endswith('}}'):
                        value = f"${{{value.lstrip('{{').rstrip('}}').rstrip(' ').lstrip(' ')}}}"
                    module_dict['environment'].update({_each: value})
            return compose_data

        except Exception as e:
            logging.exception(f'Exception occurred while process each module {e.args}')
            return False

    def process_compose_data_for_existing_files(self, docker_compose_path, variables_file, arguments,
                                                tmp_path, output_path, source_branch, destination_branch):
        docker_repo = EnvironmentVariables.docker_repo
        global_configmap = EnvironmentVariables.global_configmap
        jinja_template_file = 'docker_deployment.yml'
        template_path = "./templates"
        try:
            files_info = os.listdir(docker_compose_path)
            sorted_files = list(filter(lambda f: f.endswith(".yml") and "docker" in f, files_info))
            global_config_data = self.common_util.convert_yaml_to_define_obj(
                os.path.join(docker_compose_path, global_configmap))
            image_mapping_dict = {}
            for index, _file in enumerate(sorted_files):
                docker_compose_data = self.common_util.convert_yaml_to_define_obj(
                    os.path.join(docker_compose_path, _file))
                service_dict = docker_compose_data.get("services")
                if not service_dict:
                    logging.debug(f'Services not found for current docker compose file  - {_file}')
                if index + 1 == len(sorted_files):
                    modules = self.get_modules_dict_for_registration(modules=arguments.get('module_names', []),
                                                                     existing_modules=list(
                                                                         docker_compose_data['services'].keys()))
                    docker_compose_data['services'].update(modules)
                if response_data := self.process_module_request(compose_data=docker_compose_data,
                                                                variables_file=variables_file,
                                                                image_mapping_dict=image_mapping_dict,
                                                                general_path=tmp_path, **arguments):
                    compose_out_file_path = os.path.join(output_path, _file)
                    # self.common_util.convert_json_to_yaml(json_data=response_data,
                    #                                       output_file_path=compose_out_file_path)
                    self.common_util.render_deployment_chart(data_dict={"modules": response_data.get('services', {})},
                                                             jinja_template_file=jinja_template_file,
                                                             template_path=template_path,
                                                             outfile_path=compose_out_file_path)

        except Exception as e:
            logging.exception(f'Exception Occurred while processing the compose data info {e.args}')

    def process_compose_data_by_template(self, modules, compose_info: dict, variables_file, arguments,
                                         tmp_path, output_path):
        compose_dict = {"services": self.get_modules_dict_for_registration(modules=modules)}
        try:
            compose_out_file_path = os.path.join(output_path, "docker-compose.yml")
            jinja_template_file = 'docker_deployment.yml'
            template_path = "./templates"
            image_mapping_dict = {}
            compose_data = self.process_module_request(compose_data=compose_dict, image_mapping_dict=image_mapping_dict,
                                                       variables_file=variables_file,
                                                       general_path=tmp_path, **arguments)
            self.common_util.render_deployment_chart(data_dict={"modules": compose_data.get('services', {})},
                                                     jinja_template_file=jinja_template_file,
                                                     template_path=template_path,
                                                     outfile_path=compose_out_file_path)
        except Exception as e:
            logging.exception(f'Exception Occurred while processing the compose data by template {e.args}')

    @staticmethod
    def get_modules_dict_for_registration(modules, existing_modules=None):
        if existing_modules is None:
            existing_modules = []
        return_dict = {}
        for module in modules:
            if module not in existing_modules:
                volumes_list = [f'/data/mnt/container/logs/{module}:/code/logs',
                                '/data/mnt/container/core-volumes:/code/data']
                return_dict[module] = DockerComposeSchema(volumes=volumes_list).dict()
        return return_dict

    @staticmethod
    def get_defined_module_name(module_name, module_mapping_dict):
        for key in module_mapping_dict:
            if key.startswith(module_name):
                module_name = key
                break
        return module_name

    def get_image_tag_from_db(self, module, ilens_version, client, release_version):
        image_url = ''
        try:
            session_obj = get_db_for_func()
            module_info = self.db_handler.get_module_versions(
                input_data=GetRequest(module_name=module, client=client, ilens_version=ilens_version,
                                      release_version=release_version), db=session_obj)
            session_obj.close()
            image_url = module_info.get("image_url", '') if module_info else ''
        except Exception as e:
            logging.exception(f'Exception occurred while fetching the image details {e.args}')
        return image_url
