from fastapi.encoders import jsonable_encoder
from sqlalchemy import select, func, desc
from sqlalchemy.orm import Session

from scripts.db.psql.models import ILensVersionTable
from scripts.logging import logger
from scripts.schemas import TableInsertSchema, GetRequest
from scripts.utils.postgres_util import SQLDBUtils


class ILensVersionMaster(SQLDBUtils):
    def __init__(self, db: Session):
        super().__init__(db)
        self.table = ILensVersionTable

    def add_module_version_record(self, insert_record: TableInsertSchema):
        try:
            version_db = self.table(**insert_record.dict())
            self.session.add(version_db)
            self.session.commit()
            self.session.refresh(version_db)
            return version_db
        except Exception as e:
            logger.error(f"Could not insert a new record: {e}")
            raise

    def read_data_from_db(self, input_data: GetRequest, db: Session):
        try:
            if data := (db.query(self.table).order_by(desc(self.table.ilens_version), desc(self.table.release_version),
                                                      desc(self.table.feature_version),
                                                      desc(self.table.patch_version)).filter(
                self.table.module_name == input_data.module_name, self.table.client == input_data.client)):
                if input_data.ilens_version:
                    data = data.filter(self.table.ilens_version == input_data.ilens_version)
                if input_data.release_version:
                    data = data.filter(self.table.release_version == input_data.release_version)
                data = data.first()
                return jsonable_encoder(data)
            else:
                return None
        except Exception as e:
            logger.exception(e)
            raise

    def update_data_in_db(self, input_data: GetRequest, db: Session):
        try:
            if data := (db.query(self.table).filter(self.table.module_name == input_data.module_name,
                                                    self.table.client == input_data.client).first()):
                return jsonable_encoder(data)
            else:
                return None
        except Exception as e:
            logger.exception(e)
            raise

    def get_data_by_module_name_query(self, input_data: GetRequest, db: Session, limit_value: int = 1):
        try:
            return select(*self.table.__table__.columns,
                          func.concat(self.table.ilens_version, ".", self.table.release_version, ".",
                                      self.table.feature_version, ".", self.table.patch_version).label(
                              "module_tag")).order_by(self.table.last_updated_at).filter(
                self.table.module_name == input_data.module_name, self.table.client == input_data.client).limit(
                limit_value)

        except Exception as e:
            logger.exception(e)
            raise
